
#include "CLKernel.h"
#include "CLBuffer.h"
#include "CLImage.h"
#include "CLManager.h"
#include "CLError.h"

#include "TextFileIO.h"


namespace gk {

int CLKernel::define( const std::string& what, const std::string& value )
{
    if(m_source == NULL)
        return -1;
    return m_source->define(what, value);
}

int CLKernel::options( const std::string& options )
{
    m_options.push_back(' ');
    m_options.append(options);
    return 0;
}

int CLKernel::createCLResource( )
{
    if(make(m_source, m_name, m_options) < 0)
        printf("error building kernel '%s'.\n", m_name.c_str());
    
    return (m_kernel != 0) ? 0 : -1;
}

int CLKernel::releaseCLResource( ) 
{
    if(m_kernel == 0)
        return -1;
    
    clReleaseKernel(m_kernel);
    clReleaseProgram(m_program);
    m_program= 0;
    m_kernel= 0;
    return 0;
}
    
int CLKernel::make( const TextFile *source, const std::string& name, const std::string& options )
{
    if(source == NULL)
    {
        printf("CLKernel: make '%s': no sources.\n", name.c_str());
        return -1;
    }
    
#ifdef VERBOSE
    printf("CLKernel: make '%s', options %s\n", name.c_str(), options.c_str());
#endif
    
    // nvidia opencl ne copie pas les sources ... les conserver jusqu'a la fin de la compilation.
    std::string string(source->string());
    const char *sources= string.c_str();
    
    cl_int status= CL_SUCCESS;
    m_program= clCreateProgramWithSource(ActiveContext, 1, &sources, NULL, &status);
    if(status != CL_SUCCESS)
        return -1;

    status= clBuildProgram(m_program, 0, NULL, options.c_str(), NULL, NULL);
    if(status != CL_SUCCESS && status != CL_BUILD_PROGRAM_FAILURE)
    {
        printf("failed. %s.\n", CLError(status));
        return -1;
    }
    
    cl_build_status build= CL_BUILD_SUCCESS;
    status= clGetProgramBuildInfo(m_program, ActiveDevice, CL_PROGRAM_BUILD_STATUS, sizeof(build), &build, NULL);
    if(status != CL_SUCCESS)
        return -1;

    // afficher le log de compilation / les erreurs
    size_t length= 0;
    status= clGetProgramBuildInfo(m_program, ActiveDevice, CL_PROGRAM_BUILD_LOG, 0, NULL, &length);
    if(status != CL_SUCCESS)
    {
        printf("error building program (no info log).\n");
        return -1;
    }
    
    char *log= new char[length];
    status= clGetProgramBuildInfo(m_program, ActiveDevice, CL_PROGRAM_BUILD_LOG, length, log, NULL);
    printf("%s\ncompiler log end.\n", log);
    delete [] log;
    
    if(build != CL_BUILD_SUCCESS)
    {
        printf("failed.\n");
        return -1;
    }
    
    m_kernel= clCreateKernel(m_program, name.c_str(), &status);
    if(status != CL_SUCCESS)
        return -1;
    
    status= clGetKernelInfo(m_kernel, CL_KERNEL_NUM_ARGS, sizeof(m_parameter_count), &m_parameter_count, NULL);
    if(status != CL_SUCCESS)
        return -1;

#ifdef VERBOSE
    printf("  local memory size %dB\n", (int) localMemorySize());
    printf("  private memory size %dB\n", (int) privateMemorySize());
    printf("  workgroup max size %d\n", (int) workgroupSize());
    printf("  schedule size %d\n", (int) scheduleSize());
    printf("done.\n");
#endif
    return 0;
}

int CLKernel::setParameter( const unsigned int index, const size_t size, const void *value )
{
    if(m_kernel == 0)
        return -1;
    if(index >= m_parameter_count)
        return -1;

    //~ printf("set parameter %d, sizeof %lu, value %p\n", index, size, value);
    cl_int status= clSetKernelArg(m_kernel, index, size, value);
    if(status != CL_SUCCESS)
        printf("error: %s\n", CLError(status));
    
    return (status == CL_SUCCESS) ? 0 : -1;
}

int CLKernel::setParameter( const unsigned int index, const CLBuffer *buffer )
{
    //~ printf("set parameter %d, buffer %p\n", index, buffer->object());
    return setParameter(index, sizeof(cl_mem), buffer->object());
}

int CLKernel::setParameter( const unsigned int index, const CLImage *image )
{
    //~ printf("set parameter %d, image %p\n", index, image->object());
    return setParameter(index, sizeof(cl_mem), image->object());
}

size_t CLKernel::scheduleSize( )
{
    if(m_kernel == 0)
        return -1;

    size_t size= 0;
    cl_int status= clGetKernelWorkGroupInfo(m_kernel, ActiveDevice, CL_KERNEL_PREFERRED_WORK_GROUP_SIZE_MULTIPLE, sizeof(size), &size, NULL);
    return (status == CL_SUCCESS) ? size : 0;
}

size_t CLKernel::workgroupSize( )
{
    if(m_kernel == 0)
        return -1;

    size_t size= 0;
    cl_int status= clGetKernelWorkGroupInfo(m_kernel, ActiveDevice, CL_KERNEL_WORK_GROUP_SIZE, sizeof(size), &size, NULL);
    return (status == CL_SUCCESS) ? size : 0;
}

size_t CLKernel::CompileWorkgroupSizes( size_t sizes[3] )
{
    if(m_kernel == 0)
        return -1;

    cl_int status= clGetKernelWorkGroupInfo(m_kernel, ActiveDevice, CL_KERNEL_COMPILE_WORK_GROUP_SIZE, sizeof(sizes), sizes, NULL);
    return (status == CL_SUCCESS) ? sizes[0] : 0;
}

size_t CLKernel::CompileWorkgroupSize( )
{
    if(m_kernel == 0)
        return -1;

    size_t sizes[3];
    cl_int status= clGetKernelWorkGroupInfo(m_kernel, ActiveDevice, CL_KERNEL_COMPILE_WORK_GROUP_SIZE, sizeof(sizes), sizes, NULL);
    return (status == CL_SUCCESS) ? sizes[0] : 0;
}

size_t CLKernel::localMemorySize( )
{
    if(m_kernel == 0)
        return -1;

    cl_ulong size= 0;
    cl_int status= clGetKernelWorkGroupInfo(m_kernel, ActiveDevice, CL_KERNEL_LOCAL_MEM_SIZE, sizeof(size), &size, NULL);
    return (status == CL_SUCCESS) ? size: 0;
}

size_t CLKernel::privateMemorySize( )
{
    if(m_kernel == 0)
        return -1;

    cl_ulong size= 0;
    cl_int status= clGetKernelWorkGroupInfo(m_kernel, ActiveDevice, CL_KERNEL_PRIVATE_MEM_SIZE, sizeof(size), &size, NULL);
    return (status == CL_SUCCESS) ? size: 0;
}


CLKernel *createKernel( const std::string& filename, const std::string& name, const std::string& options )
{
#ifdef VERBOSE
    printf("loading openCL file '%s'...\n", filename.c_str());
#endif
    
    TextFile *source= TextFileIO::read(filename, name);
    if(source == NULL)
    {
    #ifdef VERBOSE
        printf("failed.\n");
    #endif
        return NULL;
    }
    
    CLKernel *kernel= new CLKernel(source, name, ActiveBuildOptions);
    if(kernel != NULL)
        kernel->options(options);
    return CLManager<CLKernel>::manager().insert(kernel, name);
}


int dispatch( CLKernel *kernel, const size_t global_size, const size_t group_size, cl_event *wait )
{
    if(ActiveQueue == NULL)
        return -1;

    return ActiveQueue->dispatch(kernel, global_size, group_size, wait);
}

int dispatch2D( CLKernel *kernel, const size_t global_width, const size_t global_height, const size_t group_width, const size_t group_height, cl_event *wait )
{
    if(ActiveQueue == NULL)
        return -1;

    return ActiveQueue->dispatch2D(kernel, global_width, global_height, group_width, group_height, wait);
}

int dispatch3D( CLKernel *kernel, cl_event *wait )
{
    if(ActiveQueue == NULL)
        return -1;
    if(kernel->object() == 0)
        return -1;

    printf("dispatch3D( ): not implemented.\n");
    return -1;
}

}
