
#include "App.h"

#include "Geometry.h"
#include "Transform.h"

#include "MeshIO.h"
#include "ShaderManager.h"
#include "setUniform.h"


class TP : public gk::App
{
    GLuint m_attributes;
    GLuint m_position_buffer;
    GLuint m_normal_buffer;
    GLuint m_index_buffer;
    int m_index_count;
    
    gk::GLShaderProgram *m_program;
    gk::Mesh *m_mesh;
    
public:
    // creation du contexte openGL et d'une fenetre
    TP( )
        :
        gk::App(),
        m_attributes(0),
        m_position_buffer(0),
        m_normal_buffer(0),
        m_index_buffer(0),
        m_index_count(0),
        m_program(NULL),
        m_mesh(NULL)
    {
        // specifie le type de contexte openGL a creer :
        gk::AppSettings settings;
        settings.setGLVersion(3,3);     // version 3.3
        settings.setGLCoreProfile();      // core profile
        settings.setGLDebugContext();     // version debug pour obtenir les messages d'erreur en cas de probleme
        
        // cree le contexte et une fenetre de dimensions 1280x768
        createWindow(1280, 768, settings);      
    }
    
    ~TP( ) {}
    
    int init( )
    {
        // etape 1 : charger l'objet
        m_mesh= gk::MeshIO::read("bigguy.obj");
        if(m_mesh == NULL)
            return -1;  // erreur de lecture
        
        // cree un shader program en compilant les shaders a partir des fichiers sources + afficher les erreurs de compilation, etc., si necessaire
        m_program= gk::createShaderProgram( "vertex_normal.vsl", "fragment_normal.fsl" );  
        if(m_program == NULL || m_program->createGLResource() < 0)
            return -1;
        
        // source du vertex shader
        /*
            uniform mat4 mvpMatrix;
            uniform mat4 normalMatrix;
            in vec3 position;                                   //!< attribut
            in vec3 normal;                                     //!< attribut
            out vec3 vertex_normal;

            void main( )
            {
                gl_Position= mvpMatrix * vec4(position, 1.0);
                vertex_normal= mat3(normalMatrix) * normal;
            }
        */
        
        // source du fragment shader
        /*
            uniform vec4 color;
            in vec3 vertex_normal;
            out vec4 fragment_color;

            void main( )
            {
                fragment_color= color * normalize(vertex_normal).z;
            }
        */
        
        // etape 1 : 
        // cree un vertex array pour conserver l'association entre les buffers et les attributs de sommets necessaires a l'execution du vertex shader
        glGenVertexArrays(1, &m_attributes);
        glBindVertexArray(m_attributes);
        
        // etape 2 : creer les buffers
        glGenBuffers(1, &m_position_buffer);
        glBindBuffer(GL_ARRAY_BUFFER, m_position_buffer);
        
        const std::vector<gk::Point>& positions= m_mesh->positions();
        glBufferData(GL_ARRAY_BUFFER, positions.size() * sizeof(gk::Point), &positions.front(), GL_STATIC_DRAW);
        {
            // recupere l'identifiant de l'attribut declare dans le vertex shader
            GLint attribut= m_program->attribute("position");
            if(attribut >= 0)
            {
                // associe le contenu du buffer courant a l'attribut de sommets necessaire a l'execution du vertex shader
                // le buffer courant est le dernier active sur GL_ARRAY_BUFFER, dans ce cas m_position_buffer
                glVertexAttribPointer(attribut, 3, GL_FLOAT, GL_FALSE, 0, 0);
                glEnableVertexAttribArray(attribut);
            }
        }
        
        glGenBuffers(1, &m_normal_buffer);
        glBindBuffer(GL_ARRAY_BUFFER, m_normal_buffer);
        
        const std::vector<gk::Normal>& normals= m_mesh->normals();
        glBufferData(GL_ARRAY_BUFFER, normals.size() * sizeof(gk::Normal), &normals.front(), GL_STATIC_DRAW);
        {
            GLint attribut= m_program->attribute("normal");
            if(attribut >= 0)
            {
                glVertexAttribPointer(attribut, 3, GL_FLOAT, GL_FALSE, 0, 0);
                glEnableVertexAttribArray(attribut);
            }
        }
        
        // index buffer
        glGenBuffers(1, &m_index_buffer);
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_index_buffer);
        
        const std::vector<int>& indices= m_mesh->indices();
        m_index_count= (int) indices.size();
        glBufferData(GL_ELEMENT_ARRAY_BUFFER, indices.size() * sizeof(int), &indices.front(), GL_STATIC_DRAW);
        
        // nettoyage, desactive les differents objets selectionnes
        glBindVertexArray(0);
        glBindBuffer(GL_ARRAY_BUFFER, 0);
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
        
        return 0;       // tout c'est bien passe, sinon renvoyer -1
    }
    
    int quit( ) 
    {
        // detruire les objets openGL crees dans init()
        glDeleteBuffers(1, &m_position_buffer);
        glDeleteBuffers(1, &m_normal_buffer);
        glDeleteBuffers(1, &m_index_buffer);
        glDeleteVertexArrays(1, &m_attributes);
        
        // le shader program et les shaders sont detruits automatiquement par gKit.
        return 0;
    }
    
    int draw( )
    {
        if(key(SDLK_ESCAPE))
            // fermer l'application si l'utilisateur appuie sur ESCAPE
            close();

        // transformations
        gk::Transform model= gk::RotateY(30.f); 
        gk::Transform view= gk::Translate( gk::Vector(0.f, 0.f, -30.f) );
        gk::Transform projection= gk::Perspective(50.f, 1.f, 1.f, 100.f);
        
        // composition des transformations
        gk::Transform mv= view * model;
        gk::Transform mvp= projection * mv;
        
        // fixer la transformation viewport en fonction des dimensions de la fenetre
        glViewport(0, 0, windowWidth(), windowHeight());
        // effacer l'image
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        
        // activer le shader
        glUseProgram(m_program->name());
        
        // parametrer le shader : matrices
        gk::setUniform(m_program->uniform("mvpMatrix"), mvp.matrix());
        gk::setUniform(m_program->uniform("normalMatrix"), mv.normalMatrix());

        gk::Color color(1.0f, 1.0f, 1.0f);
        gk::setUniform(m_program->uniform("color"), color);
        
        // selectionner / activer le vertex attrib array configure dans init( )
        glBindVertexArray(m_attributes);
        
        // dessiner
        glDrawElements(GL_TRIANGLES, m_index_count, GL_UNSIGNED_INT, 0);
        
        // nettoyage, desactive les differents objets selectionnes
        glUseProgram(0);
        glBindVertexArray(0);
        
        // afficher le buffer de dessin
        swap();
        return 1;       // continuer, dessiner une autre fois l'image, renvoyer 0 pour arreter l'application
    }
};


int main( int argc, char **argv )
{
    TP app;
    app.run();
    
    return 0;
}

