

#include "GL/GLPlatform.h"
#include "IOFileSystem.h"

#include "EffectIO.h"
#include "EffectShaderManager.h"
#include "setUniform.h"

#include "Mesh.h"
#include "MeshIO.h"
#include "BufferManager.h"

#include "SamplerManager.h"
#include "GLTexture2DIO.h"

#include "App.h"
#include "nvwidgets/nvCoreSdlWidgets.h"
#include "nvwidgets/nvSdlWidgets.h"

#include "Camera.h"
#include "Orbiter.h"


class TP : public gk::App
{
    gk::EffectShaderManager *m_effect;
    gk::GLAttributeBuffer *m_positions;
    gk::GLAttributeBuffer *m_normals;
    gk::GLAttributeBuffer *m_texcoords;
    gk::GLIndexBuffer *m_index;
    gk::GLVertexArray *m_bindings;
    gk::GLSampler *m_sampler;
    gk::Mesh *m_mesh;
    gk::GLTexture * m_default_texture;
    std::vector<gk::GLTexture *> m_diffuse_textures;
    std::vector<gk::GLTexture *> m_specular_textures;
    
    std::string m_filename;
    gk::FirstPersonCamera m_camera;
    gk::Orbiter m_orbiter;
    
    nv::CoreSdlUIContext m_ui;
    //~ nv::SdlUIContext m_ui;
    float m_scale;
    float m_mesh_scale;
    int m_wireframe;
    int m_solid;
    bool m_inspect_mode;
    int m_inspect_material;
    
public:
    TP( const std::string& filename )
        :
        gk::App(),
        m_effect(NULL),
        m_positions(NULL),
        m_normals(NULL),
        m_texcoords(NULL),
        m_index(NULL),
        m_bindings(NULL),
        m_sampler(NULL),
        m_mesh(NULL),
        m_default_texture(NULL),
        m_diffuse_textures(),
        m_specular_textures(),
        m_filename(filename),
        m_camera(),
        m_orbiter(),
        m_scale(1.f),
        m_mesh_scale(1.f),
        m_wireframe(0),
        m_solid(0),
        m_inspect_mode(0),
        m_inspect_material(-1)
    {
        gk::AppSettings settings;
        settings.setSwapControl(1);
        //~ settings.setSamples(4);
        settings.setGLVersion(3,3);
        settings.setGLCoreProfile();
        settings.setGLDebugContext();
        
        createWindow(1280, 768, settings);
    }
    
    ~TP( ) {}
    

    // a redefinir pour utiliser les widgets.
    void processWindowResize( SDL_WindowEvent& event )
    {
        m_ui.reshape(event.data1, event.data2);
    }
    
    // a redefinir pour utiliser les widgets.
    void processMouseButtonEvent( SDL_MouseButtonEvent& event )
    {
        m_ui.processMouseButtonEvent(event);
    }
    
    // a redefinir pour utiliser les widgets.
    void processMouseMotionEvent( SDL_MouseMotionEvent& event )
    {
        m_ui.processMouseMotionEvent(event);
    }
    
    // a redefinir pour utiliser les widgets.
    void processKeyboardEvent( SDL_KeyboardEvent& event )
    {
        m_ui.processKeyboardEvent(event);
    }
    
    int init( )         // charger les objets, les vertex buffers, etc.
    {
        // init widgets
        m_ui.init(windowWidth(), windowHeight());
        m_inspect_material= 0;
        
        // init camera
        m_camera= gk::FirstPersonCamera(50.f, 1.f, 1.f, 10000.f, windowWidth(), windowHeight());
        m_camera.move( gk::Point(0.f, 0.f, 50.f) );
        
        // init object manipulator
        m_orbiter= gk::Orbiter();
        
        m_effect= new gk::EffectShaderManager("viewer.gkfx");
        if(m_effect == NULL)
            return -1;
        
        gk::GLShaderProgram *program;
        program= m_effect->createShaderProgram("flat.program");
        if(program == NULL || program->createGLResource() < 0)
            return -1;
        program= m_effect->createShaderProgram("diffuse.program");
        if(program == NULL || program->createGLResource() < 0)
            return -1;
        program= m_effect->createShaderProgram("specular.program");
        if(program == NULL || program->createGLResource() < 0)
            return -1;
        
        m_mesh= gk::MeshIO::read( m_filename );
        if(m_mesh == NULL)
            return -1;
        
        gk::BBox bbox= m_mesh->bbox();
        m_mesh_scale= gk::Vector(bbox.pMin, bbox.pMax).Length();
        m_scale= 1.f;
        
        if(m_mesh->normalCount() != m_mesh->positionCount())
            m_mesh->buildNormals();
        gk::MeshIO::write(m_mesh, gk::IOFileSystem::changeType(m_filename, ".vbo.obj"));
        
        m_bindings= gk::createVertexArray();    // cree et active le vertex array
        
        // recupere les attributs necessaires a l'execution du shader et verifie que les donnees correspondantes sont bien disponibles
        m_positions= gk::createAttributeBuffer(
            m_mesh->positionCount(), 
            m_mesh->positionCount() * sizeof(float [3]), &m_mesh->positions().front());
        if(m_positions == NULL || m_positions->createGLResource() < 0)
            return -1;
        // configure le vertex array en fonction des attributs du shader
        glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, 0, 0);
        glEnableVertexAttribArray(0);
        
        if(m_mesh->normalCount() == 0)
            return -1;      // les normales sont necessaires a l'execution du shader
        
        m_normals= gk::createAttributeBuffer(
            m_mesh->normalCount(),
            m_mesh->normalCount() * sizeof(float [3]), &m_mesh->normals().front());
        if(m_normals == NULL || m_normals->createGLResource() < 0)
            return -1;
        // configure le vertex array en fonction des attributs du shader
        glVertexAttribPointer(2, 3, GL_FLOAT, GL_FALSE, 0, 0);
        glEnableVertexAttribArray(2);
    
        if(m_mesh->texcoordCount() == 0)
        {
            printf("no texcoords. failed.\n");
            return -1;      // les texcoords sont necessaires a l'execution du shader
        }
        
        // teste le type de parametrisation de la surface 
        {
            const std::vector<gk::Point2>& texcoords= m_mesh->texcoords(); 
            gk::BBox bounds;
            for(int i= 0; i < (int) texcoords.size(); i++)
            {
                gk::Point2 uv= texcoords[i];
                bounds.Union( gk::Point(uv.x, uv.y, .0f) );
            }
            
            printf("texture mapping: "); bounds.print();
        }
        
        m_texcoords= gk::createAttributeBuffer( 
            m_mesh->texcoordCount(),
            m_mesh->texcoordCount() * sizeof(float [2]), &m_mesh->texcoords().front());
        if(m_texcoords == NULL || m_texcoords->createGLResource() < 0)
            return -1;
        // configure le vertex array en fonction des attributs du shader
        glVertexAttribPointer(1, 2, GL_FLOAT, GL_FALSE, 0, 0);
        glEnableVertexAttribArray(1);
        
        if(m_mesh->indiceCount() > 0)
        {
            m_index= gk::createIndexBuffer(
                m_mesh->indiceCount(), 
                m_mesh->indiceCount() * sizeof(unsigned int), &m_mesh->indices().front());
            if(m_index == NULL || m_index->createGLResource() < 0)
                return -1;
        }
    
        // charger les textures utilisees par les matieres, si possible
        if(m_mesh->materialCount() > 0 && m_texcoords != NULL)
        {
            // recuperer les matieres utilisees par l'objet
            for(int i= 0; i < m_mesh->subMeshCount(); i++)
            {
                const gk::MeshMaterial& material= m_mesh->subMeshMaterial(i);
                
                gk::GLTexture *diffuse= NULL;
                if(material.diffuse_texture.empty() == false)
                    diffuse= gk::GLTexture2DIO::read(gk::UNIT0, material.diffuse_texture);
                // utiliser le meme indice pour la matiere et la texture correspondante
                m_diffuse_textures.push_back(diffuse);
                
                gk::GLTexture *specular= NULL;
                if(material.specular_texture.empty() == false)
                    specular= gk::GLTexture2DIO::read(gk::UNIT1, material.specular_texture);
                // utiliser le meme indice pour la matiere et la texture correspondante
                m_specular_textures.push_back(specular);
            }
        }
        
        m_default_texture= gk::GLTexture2DIO::read(gk::UNIT0, "debug_texture.png");
        if(m_default_texture == NULL)
            return -1;
        
        //~ m_sampler= gk::createAnisotropicSampler(16);
        m_sampler= gk::createLinearSampler();
        if(m_sampler == NULL || m_sampler->createGLResource() < 0)
            return -1;
        glSamplerParameteri(m_sampler->name(), GL_TEXTURE_WRAP_S, GL_REPEAT);
        glSamplerParameteri(m_sampler->name(), GL_TEXTURE_WRAP_T, GL_REPEAT);

        return 0;       // tout c'est bien passe
    }
    
    int quit( )         
    {
        delete m_effect;
        // toutes les ressources seront liberees par shaderManager et bufferManager, etc.
        return 0;
    }
    
    int draw( )
    {
        while(glGetError() != GL_NO_ERROR)
            {;}
        
        if(key(SDLK_ESCAPE))
            // sortir si l'utilisateur appuye sur ESC
            close();

        // redimensionner la fenetre, si necessaire.
        glViewport(0, 0, windowWidth(), windowHeight());
        // effacer le buffer de dessin.
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        
        // camera
        m_camera.setViewport(windowWidth(), windowHeight());
        
        // controle de la camera a la souris
        int x, y;
        int button= SDL_GetRelativeMouseState(&x, &y);
        if(button & SDL_BUTTON(1))
        {
            // move camera
            m_camera.rotateUp(-x);
            m_camera.rotateRight(-y);
        }
        if(button & SDL_BUTTON(3))
        {
            // re-orient object
            m_orbiter.rotateUp(x);
            m_orbiter.rotateRight(y);
        }
        
        // controle au clavier de la camera
        if(SDL_GetModState() & KMOD_CTRL)
        {
            if(key(SDLK_UP))
                m_camera.rotateRight(1.f);
            if(key(SDLK_DOWN))
                m_camera.rotateRight(-1.f);
            if(key(SDLK_LEFT))
                m_camera.rotateUp(1.f);
            if(key(SDLK_RIGHT))
                m_camera.rotateUp(-1.f);
        }
        else
        {
            if(key(SDLK_z) || key(SDLK_UP))
                m_camera.moveForward(-1.f);
            if(key(SDLK_s) || key(SDLK_DOWN))
                m_camera.moveForward(1.f);
            if(key(SDLK_q) || key(SDLK_LEFT))
                m_camera.moveRight(-1.f);
            if(key(SDLK_d) || key(SDLK_RIGHT))
                m_camera.moveRight(1.f);
            if(key(SDLK_PAGEUP))
                m_camera.moveUp(1.f);
            if(key(SDLK_PAGEDOWN))
                m_camera.moveUp(-1.f);
            
            if(key(SDLK_c))        // copier / coller la position de la camera
            {
                key(SDLK_c)= 0;
                m_camera.write(m_filename);
            }
            if(key(SDLK_v))
            {
                key(SDLK_v)= 0;
                m_camera.read(m_filename);
                m_orbiter.reset();
            }
            
            // modifier la vitesse de deplacement de la camera
            if(key(SDLK_KP_PLUS))
                m_camera.setSpeed(m_camera.speed() * 1.1f);
            if(key(SDLK_KP_MINUS))
                m_camera.setSpeed(m_camera.speed() * .9f);
            
            if(key(SDLK_r))
            {
                // recharger les shaders utilises
                key(SDLK_r)= 0;
                gk::IOFileManager::manager().reload(); // rescanne l'etat des fichiers 
                m_effect->reload();
            }
            
            if(key(SDLK_w))
            {
                key(SDLK_w)= 0;
                m_wireframe= (m_wireframe +1) % 2;
                
            }
            
            if(key(SDLK_x))
            {
                key(SDLK_x)= 0;
                m_solid= (m_solid +1) % 2;
            }
        }
        
        // etat openGL par defaut
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
        glDisable(GL_BLEND);
        glDisable(GL_PRIMITIVE_RESTART);

        if(m_solid == 0)
            glEnable(GL_CULL_FACE);
        else
            glDisable(GL_CULL_FACE);
        
        if(m_wireframe == 0)
        {
            glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
            glEnable(GL_CULL_FACE);
        }
        else
        {
            glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
            glDisable(GL_CULL_FACE);
        }
        
        // dessiner quelque chose.
        gk::GLShaderProgram *program= NULL;
        if(m_inspect_mode > 0 /* || m_solid > 0 */)
            program= m_effect->program("flat.program");
        else
            //~ program= m_effect->program("specular.program");
            program= m_effect->program("diffuse.program");
        
        if(program== NULL)
            return 0;
        
        glUseProgram(program->name());
        
        // calculer les transformations necessaires au shader 
        gk::Transform model=  gk::Scale(m_scale) * m_orbiter.transform();
        gk::Transform view= m_camera.view();
        gk::Transform projection= m_camera.projection();
        gk::Transform mvp= projection * view * model;
        gk::Transform mv= view * model;
        
        // passer la matrice model view projection au shader
        gk::setUniform(program->uniform("mvpMatrix"), mvp.matrix());
        gk::setUniform(program->uniform("mvMatrix"), mv.matrix());
        gk::setUniform(program->uniform("viewMatrix"), view.matrix());
        
        // parametrer les autres uniforms du shader
        gk::Color color(1.f, 1.f, 1.f);
        glBindVertexArray(m_bindings->name());
        
        if(m_index != NULL)
        {
            // draw indexe
            // parcourir les submeshes (ensembles de triangles tries par matiere)
            for(int i= 0 ; i < m_mesh->subMeshCount(); i++)
            {
                const gk::SubMesh& submesh= m_mesh->subMesh(i);
                // recuperer la matiere associee au submesh
                const gk::MeshMaterial& material= m_mesh->subMeshMaterial(i);

                // parametrer le shader
                if(m_inspect_mode > 0)
                {
                    if(i == m_inspect_material)
                        color= gk::Color(1.f, 1.f, 1.f);
                    else
                        color= gk::Color(.4, .4f, .4f);
                }
                
                gk::setUniform(program->uniform("color"), color);
                gk::setUniform(program->uniform("kd"), material.kd);
                gk::setUniform(program->uniform("ks"), material.ks);
                gk::setUniform(program->uniform("km"), material.n);
                
                // recuperer la texture diffuse associee a la matiere
                gk::GLTexture *diffuse_texture= NULL;
                if(submesh.material_id < (int) m_diffuse_textures.size())
                    diffuse_texture= m_diffuse_textures[submesh.material_id];
                if(diffuse_texture == NULL || m_solid > 0 || m_inspect_mode > 0)
                    // utiliser une texture par defaut, si la matiere n'inclut pas de texture, 
                    // ou changer de shader... 
                    diffuse_texture= m_default_texture; 
                
                // activer la texture
                glActiveTexture(GL_TEXTURE0 + gk::UNIT0);
                glBindTexture(diffuse_texture->target(), diffuse_texture->name());
                // activer le sampler (parametres de filtrage)
                glBindSampler(gk::UNIT0, m_sampler->name());
                // indiquer au shader quelle unite de texture permet d'acceder a la texture
                gk::setSamplerUniform(program->sampler("diffuse_texture"), gk::UNIT0);

                // recuperer la texture speculaire associee a la matiere
                gk::GLTexture *specular_texture= NULL;
                if(submesh.material_id < (int) m_specular_textures.size())
                    specular_texture= m_specular_textures[submesh.material_id];
                if(specular_texture == NULL || m_solid > 0 || m_inspect_mode > 0)
                    // utiliser une texture par defaut, si la matiere n'inclut pas de texture, 
                    // ou changer de shader... 
                    specular_texture= m_default_texture; 
                
                // activer la texture
                glActiveTexture(GL_TEXTURE0 + gk::UNIT1);
                glBindTexture(specular_texture->target(), specular_texture->name());
                // activer le sampler (parametres de filtrage)
                glBindSampler(gk::UNIT1, m_sampler->name());
                // indiquer au shader quelle unite de texture permet d'acceder a la texture
                gk::setSamplerUniform(program->sampler("specular_texture"), gk::UNIT1);
            
                // dessine les triangles associes au submesh
                int count= submesh.end - submesh.begin;
                long int offset= submesh.begin * sizeof(unsigned int);
                glDrawElements(GL_TRIANGLES, count, GL_UNSIGNED_INT, (const GLvoid *) offset);
                // le dernier parametre est un 'offset' dans le buffer d'indices, en octets
                
                // remarque : utiliser la classe gk::BufferLayout pour stocker l'organisation des donnees dans un buffer et la methode
                // gk::BufferLayout::getOffset( ) 
            }
        }
        else
        {
            // pas d'index, donc pas de submesh, tout dessiner... des points ?
            glDrawArrays(GL_POINTS, 0, m_positions->count());
        }

        // nettoyer
        glBindSampler(gk::UNIT0, 0);
        glBindSampler(gk::UNIT1, 0);
        glActiveTexture(GL_TEXTURE0 + gk::UNIT1);
        glBindTexture(GL_TEXTURE_2D, 0);
        glActiveTexture(GL_TEXTURE0 + gk::UNIT0);
        glBindTexture(GL_TEXTURE_2D, 0);
        
        glBindVertexArray(0);
        glUseProgram(0);
        
    #if 1
        // gui
        m_ui.begin();
        m_ui.beginGroup(nv::GroupFlags_GrowDownFromLeft);
        
        char tmp[1024];
        m_ui.beginGroup(nv::GroupFlags_GrowRightFromTop);
            snprintf(tmp, sizeof(tmp), "scale (%f) %f", m_mesh_scale, m_scale);
            m_ui.doLabel(nv::Rect(), tmp);
            m_ui.doHorizontalSlider(nv::Rect(0, 0, 400, 0), 0.f, 10.f, &m_scale);
        m_ui.endGroup();
        
        if(m_ui.doButton(nv::Rect(), "inspect mesh", &m_inspect_mode))
        {
            if(m_inspect_material < 0)
                m_inspect_material= 0;
        }
        
        if(m_inspect_mode > 0)
        {
            std::vector<const char *> materials;
            for(int i= 0; i < m_mesh->subMeshCount(); i++)
                materials.push_back(m_mesh->subMeshMaterial(i).name.c_str());
            
            m_ui.doComboBox(nv::Rect(), materials.size(), &materials.front(), &m_inspect_material);
            if(m_inspect_material > -1)
            {
                // affiche les parametres de la matiere selectionnee
                const gk::MeshMaterial& material= m_mesh->subMeshMaterial(m_inspect_material);
                
                m_ui.doLabel(nv::Rect(), material.name.c_str());
                
                m_ui.beginGroup(nv::GroupFlags_GrowRightFromTop);
                    m_ui.doLabel(nv::Rect(), "Kd");
                    gk::Color kd= material.diffuse;
                    snprintf(tmp, sizeof(tmp), "(%f) %f %f %f", material.kd, kd.r, kd.g, kd.b);
                    m_ui.doLabel(nv::Rect(), tmp);
                    //~ m_ui.doLineEdit();
                    
                    snprintf(tmp, sizeof(tmp), "map_Kd (%s) %s", m_diffuse_textures[m_inspect_material] ? "loaded" : "debug", material.diffuse_texture.c_str());
                    m_ui.doLabel(nv::Rect(), tmp);
                m_ui.endGroup();
                
                m_ui.beginGroup(nv::GroupFlags_GrowRightFromTop);
                    m_ui.doLabel(nv::Rect(), "Ks");
                    //~ m_ui.doLineEdit();
                    gk::Color ks= material.specular;
                    snprintf(tmp, sizeof(tmp), "(%f) %f %f %f", material.ks, ks.r, ks.g, ks.b);
                    m_ui.doLabel(nv::Rect(), tmp);
                    snprintf(tmp, sizeof(tmp), "map_Ks (%s) %s", m_specular_textures[m_inspect_material] ? "loaded" : "debug", material.specular_texture.c_str());
                    m_ui.doLabel(nv::Rect(), tmp);
                m_ui.endGroup();
                
                m_ui.beginGroup(nv::GroupFlags_GrowRightFromTop);
                    m_ui.doLabel(nv::Rect(), "Km");
                    //~ m_ui.doLineEdit();
                    float km= material.n;
                    snprintf(tmp, sizeof(tmp), "%f", km);
                    m_ui.doLabel(nv::Rect(), tmp);
                m_ui.endGroup();
            }
        }
        else
        {
            m_inspect_material= -1;
        }
        
        m_ui.endGroup();
        m_ui.end();
    #endif
    
        // afficher.
        swap();
        
        // continuer.
        return 1;
    }
};


int main( int argc, char *argv[] )
{
    std::string filename= "bigguy.obj";
    if(argc > 1)
        filename= argv[1];
    
    TP app(filename);
    app.run();
    
    return 0;
}
