
#ifndef _GLCore_DRAW_H
#define _GLCore_DRAW_H

#include <vector>
#include <map>

#include <cstdio>


namespace nv {
namespace GLCore {

//! vertex data: 2d position, 3d texcoords.
struct vertex
{
    float x, y;
    float s, t, p;
    
    vertex( )
        :
        x(0.f), y(0.f),
        s(0.f), t(0.f), p(0.f)
    {}
    
    vertex( const float _x, const float _y, const float _s, const float _t, const float _p= 0.f )
        :
        x(_x), y(_y),
        s(_s), t(_t), p(_p)
    {}
};

//! generic vec2.
struct vec2
{
    float x, y;
    
    vec2( )
        :
        x(0.f), y(0.f)
    {}
    
    vec2( const float _x, const float _y )
        :
        x(_x), y(_y)
    {}
    
    vec2& operator=( const vec2& b )
    {
        x= b.x;
        y= b.y;
        return *this;
    }
    
    float operator[] ( const unsigned int k ) const
    {
        return (&x)[k];
    }
    
    float& operator[] ( const unsigned int k ) 
    {
        return (&x)[k];
    }
    
    operator float *( )
    {
        return &x;
    }
    
    operator const float *( ) const
    {
        return &x;
    }
};

//! generic vec4.
struct vec4
{
    float x, y, z, w;
    
    vec4( )
        :
        x(0.f), y(0.f), z(0.f), w(0.f)
    {}
    
    vec4( const float _x, const float _y, const float _z, const float _w )
        :
        x(_x), y(_y), z(_z), w(_w)
    {}
    
    vec4& operator=( const vec4& b )
    {
        x= b.x;
        y= b.y;
        z= b.z;
        w= b.w;
        return *this;
    }
    
    float operator[] ( const unsigned int k ) const
    {
        return (&x)[k];
    }
    
    float& operator[] ( const unsigned int k ) 
    {
        return (&x)[k];
    }
    
    operator float *( )
    {
        return &x;
    }
    
    operator const float *( ) const
    {
        return &x;
    }
};

//! generic mat4.
struct matrix
{
    float m[4][4];
    
    matrix( ) {}
    matrix(
        float t00, float t01, float t02, float t03,
        float t10, float t11, float t12, float t13,
        float t20, float t21, float t22, float t23,
        float t30, float t31, float t32, float t33)
    {
        m[0][0] = t00; m[0][1] = t01; m[0][2] = t02; m[0][3] = t03;
        m[1][0] = t10; m[1][1] = t11; m[1][2] = t12; m[1][3] = t13;
        m[2][0] = t20; m[2][1] = t21; m[2][2] = t22; m[2][3] = t23;
        m[3][0] = t30; m[3][1] = t31; m[3][2] = t32; m[3][3] = t33;
    }
};

//! required state for drawing something with Draw<>
struct params
{
    unsigned int start;
    unsigned int end;
    
    params( )
        :
        start(0),
        end(0)
    {}
};


//! basic batch with immediate mode vertex submission (glBegin / glEnd).
template < class P >
class Draw
{
protected:
    GLuint m_bindings;
    GLuint m_buffer;
    GLuint m_index_buffer;
    matrix m_projection;
    
    vertex m_vertex;

    std::vector<vertex> m_data;
    std::vector<unsigned int> m_index;
    std::vector<P> m_draws;
    //~ std::map< P, int > m_states;
    
public:
    Draw( )
        :
        m_bindings(0),
        m_buffer(0),
        m_index_buffer(0),
        m_vertex()
    {}
        
    void init( )
    {
        glGenVertexArrays(1, &m_bindings);
        glBindVertexArray(m_bindings);
        
        glGenBuffers(1, &m_buffer);
        glBindBuffer(GL_ARRAY_BUFFER, m_buffer);
        glBufferData(GL_ARRAY_BUFFER, 8192, NULL, GL_STREAM_DRAW);
        
        glGenBuffers(1, &m_index_buffer);
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_index_buffer);
        glBufferData(GL_ELEMENT_ARRAY_BUFFER, 8192, NULL, GL_STREAM_DRAW);
        
        // position
        glVertexAttribPointer(0, 2, GL_FLOAT, GL_FALSE, sizeof(vertex), (const GLvoid *) 0);
        glEnableVertexAttribArray(0);
        // texcoord
        glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, sizeof(vertex), (const GLvoid *) sizeof(float [2]));
        glEnableVertexAttribArray(1);

        glBindVertexArray(0);
        glBindBuffer(GL_ARRAY_BUFFER, 0);
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
    }
    
    ~Draw( )
    {
        glDeleteBuffers(1, &m_buffer);
        glDeleteBuffers(1, &m_index_buffer);
        glDeleteVertexArrays(1, &m_bindings);
    }

    void begin( const P& params )
    {
        m_vertex= vertex();
        
        // sort states
        //~ std::pair< std::map< P, int >::iterator, bool > insert= m_states.insert( 
            //~ std::make_pair(params, m_draws.size()) );
        //~ if(insert.second == true)
            //~ m_draws.push_back( std::vector<P>(1, params) );
        
        m_draws.push_back( params );
        m_draws.back().start= m_index.size();
    }
    
    void end( )
    {
        m_index.push_back(0xffff);    // restart index 
        m_draws.back().end= m_index.size();
    }

    void clear( )
    {
        m_data.clear();
        m_index.clear();
        m_draws.clear();
    }
    
    void reshape( const Rect& window )
    {
        const float left= window.x;
        const float right= window.w;
        const float bottom= window.y;
        const float top= window.h;
        const float znear= -1.f;
        const float zfar= 1.f;
        
        m_projection= matrix(
            2.f / (right - left)     , 0.f                 , 0.f                  , -(right + left) / (right - left),
            0.f                      , 2.f / (top - bottom), 0.f                  , -(top + bottom) / (top - bottom),
            0.f                      , 0.f                 , -2.f / (zfar - znear), -(zfar + znear) / (zfar - znear),
            0.f, 0.f, 0.f, 1.f);
    }
    
    void push_texcoord( const float s, const float t, const float p= 0.f )
    {
        m_vertex.s= s;
        m_vertex.t= t;
        m_vertex.p= p;
    }
    
    void push_vertex( const float x, const float y )
    {
        m_vertex.x= x;
        m_vertex.y= y;
        
        m_index.push_back( m_data.size() );
        m_data.push_back( m_vertex );
    }
    
    void push_vertex( const float x, const float y, const float s, const float t, const float p= 0.f )
    {
        push_texcoord(s, t, p);
        push_vertex(x, y);
    }
    
    void restart( )
    {
        m_index.push_back(0xffff);    // restart index 
    }
    
    void draw( )
    {
        if(m_draws.back().end == 0)
            end();
        
        // resize vertex buffer when necessary
        GLint buffer_size;
        glBindBuffer(GL_ARRAY_BUFFER, m_buffer);
        glGetBufferParameteriv(GL_ARRAY_BUFFER, GL_BUFFER_SIZE, &buffer_size);
        if((size_t) buffer_size < m_data.size() * sizeof(vertex))
            glBufferData(GL_ARRAY_BUFFER, m_data.capacity() * sizeof(vertex), NULL, GL_STREAM_DRAW);
        
        // update vertex buffer
        glBufferSubData(GL_ARRAY_BUFFER, 0, m_data.size() * sizeof(vertex), (const GLvoid *) &m_data.front());
        
        // resize index buffer when necessary
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_index_buffer);
        glGetBufferParameteriv(GL_ELEMENT_ARRAY_BUFFER, GL_BUFFER_SIZE, &buffer_size);
        if((size_t) buffer_size < m_index.size() * sizeof(unsigned int))
            glBufferData(GL_ELEMENT_ARRAY_BUFFER, m_index.capacity() * sizeof(unsigned int), NULL, GL_STREAM_DRAW);
        
        // update index buffer
        glBufferSubData(GL_ELEMENT_ARRAY_BUFFER, 0, m_index.size() * sizeof(unsigned int), (const GLvoid *) &m_index.front());

        // draw states
        glPrimitiveRestartIndex(0xffff);
        glEnable(GL_PRIMITIVE_RESTART);
        
        glBindVertexArray(m_bindings);
        const void *program= NULL;
        
        int count= m_draws.size();
        for(int i= 0; i < count; i++)
        {
            // bind shader
            if(m_draws[i].program != program)
            {
                glUseProgram(m_draws[i].program->program);
                // update projection matrix
                assert(m_draws[i].program->projection > -1);
                glUniformMatrix4fv(m_draws[i].program->projection, 1, GL_TRUE, (const GLfloat *) m_projection.m);
                
                program= m_draws[i].program;
            }
            
            // set uniforms
            m_draws[i].apply();
            
            // draw strips
            glDrawElements(GL_TRIANGLE_STRIP, 
                m_draws[i].end - m_draws[i].start, 
                GL_UNSIGNED_INT, (const GLvoid *) (m_draws[i].start * sizeof(unsigned int)));
        }
        
        glBindVertexArray(0);
        glBindBuffer(GL_ARRAY_BUFFER, 0);
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
    }
};

}       // namespace GLCore
}       // namespace nv
#endif
