
#include "nvCoreSdlTextPainter.h"

using namespace nv;


#include <cassert>
#include <ctype.h>
#include <cmath>

#define NV_REPORT_COMPILE_ERRORS
#include <nvglutils/nvShaderUtils.h>

#include "nvCoreSdlTextPainter.h"
#include "nvGLCoreDraw.h"


namespace nv {
    
CoreSdlTextPainter::CoreSdlTextPainter( const std::string& filename, const int size )
    :
    SdlTextPainter(filename, size)
{}
    
CoreSdlTextPainter::~CoreSdlTextPainter( )
{
    glDeleteProgram(m_text_program.program);
    glDeleteSamplers(1, &m_sampler);
}

static
const char *text_vertex= {
"   #version 330    // core vertex shader\n\
    uniform mat4 projection;\n\
    layout(location= 0) in vec2 position;\n\
    layout(location= 1) in vec3 texcoord;\n\
    out vec3 vertex_texcoord;\n\
    void main( )\n\
    {\n\
        vertex_texcoord= texcoord;\n\
        gl_Position= projection * vec4(position, 0.0, 1.0);\n\
    }\n\
"
};

static
const char *text_fragment= {
"    #version 330    // core fragment shader\n\
    uniform vec4 color;\n\
    uniform sampler2D font;\n\
    in vec3 vertex_texcoord;\n\
    out vec4 fragment_color;\n\
    void main( )\n\
    {\n\
        fragment_color= color * texture(font, vertex_texcoord.st);\n\
    }\n\
"
};

void CoreSdlTextPainter::init( )
{
    GLuint vertex= nv::CompileGLSLShader(GL_VERTEX_SHADER, text_vertex);
    GLuint fragment= nv::CompileGLSLShader(GL_FRAGMENT_SHADER, text_fragment);
    GLuint program= nv::LinkGLSLProgram(vertex, fragment);
    if(program == 0)
        return;
    assert(glGetError() == GL_NO_ERROR);
    
    m_text_program.program= program;
    m_text_program.projection= glGetUniformLocation(program, "projection");
    m_text_program.color= glGetUniformLocation(program, "color");
    m_text_program.font= glGetUniformLocation(program, "font");
    assert(glGetError() == GL_NO_ERROR);
    
    glGenSamplers(1, &m_sampler);
    glBindSampler(0, m_sampler);
    glSamplerParameteri(m_sampler, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glSamplerParameteri(m_sampler, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
    glSamplerParameteri(m_sampler, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glSamplerParameteri(m_sampler, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glBindSampler(0, 0);

    SDL_Surface *surface= init_glyph_cache();
    
    glGenTextures(1, &m_texture);
    assert(m_texture != 0);
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, m_texture);

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
    glTexImage2D(GL_TEXTURE_2D, 
        0, GL_RGBA, surface->w, surface->h, 0,
        GL_RGBA, GL_UNSIGNED_BYTE, surface->pixels);
    glGenerateMipmap(GL_TEXTURE_2D);
    SDL_FreeSurface(surface);

    m_texts.init();
    m_is_init= true;
}

void CoreSdlTextPainter::begin( const Rect& window )
{
    m_texts.reshape(window);
}

void CoreSdlTextPainter::end( )
{
    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, m_texture);
    glBindSampler(0, m_sampler);

    m_texts.draw();
    m_texts.clear();
    
    glBindTexture(GL_TEXTURE_2D, 0);
    glBindSampler(0, 0);
    
    //~ {
        //~ GLint unit= -1;
        //~ glGetIntegerv(GL_ACTIVE_TEXTURE, &unit);
        //~ GLint name= 0;
        //~ glGetIntegerv(GL_TEXTURE_BINDING_2D, &name);
        //~ GLint sampler= 0;
        //~ glGetIntegerv(GL_SAMPLER_BINDING, &sampler);
        //~ printf("after: active texture unit %x, texture %d, sampler %d\n", unit, name, sampler);
    //~ }
}

//~ void CoreSdlTextPainter::drawCharacter( int x, int y, const char c )
//~ {
    //~ return;
//~ }

void CoreSdlTextPainter::draw_glyph( const int x, const int y, const int id )
{
    assert(id >= 0 && id < 128);
    
    // align top left corner 
    // cf. http://www.libsdl.org/projects/SDL_ttf/docs/SDL_ttf_38.html#SEC38
    const float yy= y +2;
    
    m_texts.push_texcoord(m_cache[id].tex_xmin, m_cache[id].tex_ymin);
    m_texts.push_vertex(x + m_cache[id].xmin, yy + m_cache[id].ymin);
    
    m_texts.push_texcoord(m_cache[id].tex_xmax, m_cache[id].tex_ymin);
    m_texts.push_vertex(x + m_cache[id].xmax, yy + m_cache[id].ymin);
    
    m_texts.push_texcoord(m_cache[id].tex_xmin, m_cache[id].tex_ymax);
    m_texts.push_vertex(x + m_cache[id].xmin, yy + m_cache[id].ymax);
    
    m_texts.push_texcoord(m_cache[id].tex_xmax, m_cache[id].tex_ymax);
    m_texts.push_vertex(x + m_cache[id].xmax, yy + m_cache[id].ymax);
    
    m_texts.restart();
}

void CoreSdlTextPainter::drawString( int x, int y, const char *text, int lines, int colorId, const float *color )
{
    nv::GLCore::text_params params(m_text_program);
    params.colorId= colorId;
    params.color= nv::GLCore::vec4(color[0], color[1], color[2], color[3]);
    params.font= 0;     // texture unit
    
    m_texts.begin(params);
    
    int w= 0;
    int n= 0;
    for(int i= 0; n < lines && text[i] != 0; i++)
    {
        if(text[i] == '\n')
        {
            n+= 1;
            w= 0;
            continue;
        }
        
        const int g= ((unsigned char) text[i] < 128) ? (unsigned char) text[i] : '?';
        draw_glyph(x + w, y + m_line_skip * n, g);
        
        w+= m_cache[g].advance;
    }
    
    m_texts.end();
}

}
