

#include <SDL2/SDL.h>

#define NV_REPORT_COMPILE_ERRORS
#include <../nvglutils/nvShaderUtils.h>

#include "nvGLWidgets.h"
#include "nvGLCoreDraw.h"
#include "nvGLCoreUIPainter.h"


SDL_Window *window= NULL;
SDL_GLContext gl_context= NULL;

unsigned int context_flags= SDL_GL_CONTEXT_DEBUG_FLAG;
unsigned int profile_flags= SDL_GL_CONTEXT_PROFILE_CORE;

int width= 1024;
int height= 768;

nv::GLCore::widget_program widget_program;
nv::GLCore::Draw<nv::GLCore::widget_params> widgets;


#define norm255( i ) ( (float) ( i ) / 255.0f )

enum Color {
    cBase = 0,
    cBool = 4,
    cOutline = 8,
    cFont = 12,
    cFontBack = 16,
    cTranslucent = 20,
    cNbColors = 24,
};

const static nv::GLCore::vec4 s_colors[cNbColors] = {
    // cBase
    nv::GLCore::vec4( norm255(89), norm255(89), norm255(89), 0.7f ),
    nv::GLCore::vec4( norm255(166), norm255(166), norm255(166), 0.8f ),
    nv::GLCore::vec4( norm255(212), norm255(228), norm255(60), 0.5f ),
    nv::GLCore::vec4( norm255(227), norm255(237), norm255(127), 0.5f ),

    // cBool
    nv::GLCore::vec4( norm255(99), norm255(37), norm255(35), 1.0f ),
    nv::GLCore::vec4( norm255(149), norm255(55), norm255(53), 1.0f ),
    nv::GLCore::vec4( norm255(212), norm255(228), norm255(60), 1.0f ),
    nv::GLCore::vec4( norm255(227), norm255(237), norm255(127), 1.0f ),

    // cOutline
    nv::GLCore::vec4( norm255(255), norm255(255), norm255(255), 1.0f ),
    nv::GLCore::vec4( norm255(255), norm255(255), norm255(255), 1.0f ),
    nv::GLCore::vec4( norm255(255), norm255(255), norm255(255), 1.0f ),
    nv::GLCore::vec4( norm255(255), norm255(255), norm255(255), 1.0f ),

    // cFont
    nv::GLCore::vec4( norm255(255), norm255(255), norm255(255), 1.0f ),
    nv::GLCore::vec4( norm255(255), norm255(255), norm255(255), 1.0f ),
    nv::GLCore::vec4( norm255(255), norm255(255), norm255(255), 1.0f ),
    nv::GLCore::vec4( norm255(255), norm255(255), norm255(255), 1.0f ),

    // cFontBack
    nv::GLCore::vec4( norm255(79), norm255(129), norm255(189), 1.0 ),
    nv::GLCore::vec4( norm255(79), norm255(129), norm255(189), 1.0 ),
    nv::GLCore::vec4( norm255(128), norm255(100), norm255(162), 1.0 ),
    nv::GLCore::vec4( norm255(128), norm255(100), norm255(162), 1.0 ),

    // cTranslucent
    nv::GLCore::vec4( norm255(0), norm255(0), norm255(0), 0.0 ),
    nv::GLCore::vec4( norm255(0), norm255(0), norm255(0), 0.0 ),
    nv::GLCore::vec4( norm255(0), norm255(0), norm255(0), 0.0 ),
    nv::GLCore::vec4( norm255(0), norm255(0), norm255(0), 0.0 ),
};



void debug( GLenum source,  GLenum type, unsigned int id, GLenum severity, GLsizei length, const char* message, void* userParam )
{
    printf("opengl message: %s\n", message);
    fflush(stdout);
}


int init( )
{
    GLuint vertex= nv::CompileGLSLShaderFromFile(GL_VERTEX_SHADER, "widget.vsl");
    GLuint fragment= nv::CompileGLSLShaderFromFile(GL_FRAGMENT_SHADER, "widget.fsl");
    GLuint program= nv::LinkGLSLProgram(vertex, fragment);
    if(program == 0)
        return -1;
    assert(glGetError() == GL_NO_ERROR);
    
    widget_program.program= program;
    widget_program.projection= glGetUniformLocation(program, "projection");
    widget_program.fillColor= glGetUniformLocation(program, "fillColor");
    widget_program.borderColor= glGetUniformLocation(program, "borderColor");
    widget_program.zones= glGetUniformLocation(program, "zones");
    assert(glGetError() == GL_NO_ERROR);
    
    widgets.init();
    assert(glGetError() == GL_NO_ERROR);
    return 0;
}


void drawRect( const nv::Rect & rect, int fillColorId, int borderColorId ) 
{
    nv::GLCore::widget_params params(widget_program);
    params.borderId= borderColorId;
    params.border= s_colors[borderColorId];
    params.fillId= fillColorId;
    params.fill= s_colors[fillColorId];
    params.zones= nv::GLCore::vec2(0, 0);
    
    float x0 = rect.x;
    float x1 = rect.x + rect.w;

    float y0 = rect.y;
    float y1 = rect.y + rect.h;

    widgets.begin(params);
        widgets.push_vertex(x0, y0, 0, 0);
        widgets.push_vertex(x1, y0, 0, 0);
        widgets.push_vertex(x0, y1, 0, 0);
        widgets.push_vertex(x1, y1, 0, 0);
    widgets.end();

#if 0
    glUseProgram(m_widgetProgram);

    glUniform4fv( m_fillColorUniform, 1, s_colors[fillColorId]);
    glUniform4fv( m_borderColorUniform, 1, s_colors[borderColorId]);
    glUniform2f( m_zonesUniform, 0, 0);

    printf("draw rect fill %d border %d, zone %f %f\n", fillColorId, borderColorId, 0.f, 0.f);

    float x0 = rect.x;
    float x1 = rect.x + rect.w;

    float y0 = rect.y;
    float y1 = rect.y + rect.h;

    glBegin(GL_TRIANGLE_STRIP);
    glTexCoord2f(0, 0);
    glVertex2f( x0, y0);

    glTexCoord2f(0, 0);
    glVertex2f( x1, y0);

    glTexCoord2f(0, 0);
    glVertex2f( x0, y1);

    glTexCoord2f(0, 0);
    glVertex2f( x1, y1);
    glEnd();

    glUseProgram(0);
#endif
}


void drawRoundedRect( const nv::Rect& rect, const nv::Point& corner, int fillColorId, int borderColorId )
{
    nv::GLCore::widget_params params(widget_program);
    params.borderId= borderColorId;
    params.border= s_colors[borderColorId];
    params.fillId= fillColorId;
    params.fill= s_colors[fillColorId];
    params.zones= nv::GLCore::vec2(corner.x - 1, corner.x - 2);

    //~ glUseProgram(m_widgetProgram);
    //~ glUniform4fv( m_fillColorUniform, 1, s_colors[fillColorId]);
    //~ glUniform4fv( m_borderColorUniform, 1, s_colors[borderColorId]);
    //~ glUniform2f( m_zonesUniform, corner.x - 1, corner.x - 2);

    //~ printf("draw rounded rect fill %d border %d, zone %f %f\n", fillColorId, borderColorId, corner.x - 1.f, corner.x - 2.f);
    
    float xb = corner.x;
    float yb = corner.y;
    
    float x0 = rect.x;
    float x1 = rect.x + corner.x;
    float x2 = rect.x + rect.w - corner.x;
    float x3 = rect.x + rect.w;

    float y0 = rect.y;
    float y1 = rect.y + corner.y;
    float y2 = rect.y + rect.h - corner.y;
    float y3 = rect.y + rect.h;

    //~ glBegin(GL_TRIANGLE_STRIP);
    widgets.begin(params);
    //~ glTexCoord2f(xb, yb);
    widgets.push_vertex( x0, y0, xb, yb );
    //~ glTexCoord2f(0, yb);
    widgets.push_vertex( x1, y0, 0, yb );

    //~ glTexCoord2f(xb, 0);
    widgets.push_vertex( x0, y1, xb, 0 );
    //~ glTexCoord2f(0, 0);
    widgets.push_vertex( x1, y1, 0, 0 );

    //~ glTexCoord2f(xb, 0);
    widgets.push_vertex( x0, y2, xb, 0 );
    //~ glTexCoord2f(0, 0);
    widgets.push_vertex( x1, y2, 0, 0 );

    //~ glTexCoord2f(xb, yb);
    widgets.push_vertex( x0, y3, xb, yb );
    //~ glTexCoord2f(0, yb);
    widgets.push_vertex( x1, y3, 0, yb);
    
    //~ glEnd();
    //~ glBegin(GL_TRIANGLE_STRIP);
    widgets.restart();
    //~ widgets.end();
    //~ widgets.begin(params);
    
    //~ glTexCoord2f(0, yb);
    widgets.push_vertex( x2, y0, 0, yb );
    //~ glTexCoord2f(xb, yb);
    widgets.push_vertex( x3, y0, xb, yb );

    //~ glTexCoord2f(0, 0);
    widgets.push_vertex( x2, y1, 0, 0 );
    //~ glTexCoord2f(xb, 0);
    widgets.push_vertex( x3, y1, 0, 0 );

    //~ glTexCoord2f(0, 0);
    widgets.push_vertex( x2, y2, 0, 0);
    //~ glTexCoord2f(xb, 0);
    widgets.push_vertex( x3, y2, xb, 0 );

    //~ glTexCoord2f(0, yb);
    widgets.push_vertex( x2, y3, 0, yb);
    //~ glTexCoord2f(xb, yb);
    widgets.push_vertex( x3, y3, xb, yb );
    //~ glEnd();
    //~ glBegin(GL_TRIANGLE_STRIP);
    //~ widgets.restart();
    widgets.end();
    //~ widgets.begin(params);
    
    //~ glTexCoord2f(0, yb);
    widgets.push_vertex( x1, y0, 0, yb );
    //~ glTexCoord2f(0, yb);
    widgets.push_vertex( x2, y0, 0, yb );

    //~ glTexCoord2f(0, 0);
    widgets.push_vertex( x1, y1, 0, 0 );
    //~ glTexCoord2f(0, 0);
    widgets.push_vertex( x2, y1, 0, 0 );

    //~ glTexCoord2f(0, 0);
    widgets.push_vertex( x1, y2, 0, 0 );
    //~ glTexCoord2f(0, 0);
    widgets.push_vertex( x2, y2, 0, 0 );

    //~ glTexCoord2f(0, yb);
    widgets.push_vertex( x1, y3, 0, yb );
    //~ glTexCoord2f(0, yb);
    widgets.push_vertex( x2, y3, 0, yb );
    //~ glEnd();
    //~ glUseProgram(0);
    widgets.end();
}

#if 0
void drawRoundedRectOutline( const nv::Rect& rect, const nv::Point& corner, int borderColorId ) 
{
    glUseProgram(m_widgetProgram);
    glUniform4fv( m_fillColorUniform, 1, s_colors[cTranslucent]);
    glUniform4fv( m_borderColorUniform, 1, s_colors[borderColorId]);
    glUniform2f( m_zonesUniform, corner.x - 1, corner.x - 2);

    printf("draw rounded rect outline fill %d border %d, zone %f %f\n", cTranslucent, borderColorId, corner.x - 1.f, corner.x - 2.f);
    
    float xb = corner.x;
    float yb = corner.y;

    float x0 = rect.x;
    float x1 = rect.x + corner.x;
    float x2 = rect.x + rect.w - corner.x;
    float x3 = rect.x + rect.w;

    float y0 = rect.y;
    float y1 = rect.y + corner.y;
    float y2 = rect.y + rect.h - corner.y;
    float y3 = rect.y + rect.h;

    glBegin(GL_TRIANGLE_STRIP);
    glTexCoord2f(xb, yb);
    glVertex2f( x0, y0);
    glTexCoord2f(0, yb);
    glVertex2f( x1, y0);

    glTexCoord2f(xb, 0);
    glVertex2f( x0, y1);
    glTexCoord2f(0, 0);
    glVertex2f( x1, y1);

    glTexCoord2f(xb, 0);
    glVertex2f( x0, y2);
    glTexCoord2f(0, 0);
    glVertex2f( x1, y2);

    glTexCoord2f(xb, yb);
    glVertex2f( x0, y3);
    glTexCoord2f(0, yb);
    glVertex2f( x1, y3);
    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glTexCoord2f(0, yb);
    glVertex2f( x2, y0);
    glTexCoord2f(xb, yb);
    glVertex2f( x3, y0);

    glTexCoord2f(0, 0);
    glVertex2f( x2, y1);
    glTexCoord2f(xb, 0);
    glVertex2f( x3, y1);

    glTexCoord2f(0, 0);
    glVertex2f( x2, y2);
    glTexCoord2f(xb, 0);
    glVertex2f( x3, y2);

    glTexCoord2f(0, yb);
    glVertex2f( x2, y3);
    glTexCoord2f(xb, yb);
    glVertex2f( x3, y3);
    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glTexCoord2f(0, yb);
    glVertex2f( x1, y0);
    glTexCoord2f(0, yb);
    glVertex2f( x2, y0);

    glTexCoord2f(0, 0);
    glVertex2f( x1, y1);
    glTexCoord2f(0, 0);
    glVertex2f( x2, y1);
    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glTexCoord2f(0, 0);
    glVertex2f( x1, y2);
    glTexCoord2f(0, 0);
    glVertex2f( x2, y2);

    glTexCoord2f(0, yb);
    glVertex2f( x1, y3);
    glTexCoord2f(0, yb);
    glVertex2f( x2, y3);
    glEnd();

    glUseProgram(0);
}

void drawCircle( const nv::Rect& rect, int fillColorId, int borderColorId ) 
{
    glUseProgram(m_widgetProgram);

    glUniform4fv( m_fillColorUniform, 1, s_colors[fillColorId]);
    glUniform4fv( m_borderColorUniform, 1, s_colors[borderColorId]);
    glUniform2f( m_zonesUniform, (rect.w / 2) - 1, (rect.w / 2) - 2);

    printf("draw circle fill %d border %d, zone %f %f\n", fillColorId, borderColorId, (rect.w / 2) - 1.f, (rect.w / 2) - 2.f);

    float xb = rect.w / 2;
    float yb = rect.w / 2;

    float x0 = rect.x;
    float x1 = rect.x + rect.w;

    float y0 = rect.y;
    float y1 = rect.y + rect.h;

    glBegin(GL_TRIANGLE_STRIP);
    glTexCoord2f(-xb, -yb);
    glVertex2f( x0, y0);
    glTexCoord2f(xb, -yb);
    glVertex2f( x1, y0);
    glTexCoord2f(-xb, yb);
    glVertex2f( x0, y1);
    glTexCoord2f(xb, yb);
    glVertex2f( x1, y1);
    glEnd();

    glUseProgram(0);
}

void drawMinus( const nv::Rect& rect, int width, int fillColorId, int borderColorId )
{
    float xb = width;
    float yb = width;

    float xoff = xb ;
    float yoff = yb ;

    float x0 = rect.x + rect.w * 0.1 ;
    float x1 = rect.x + rect.w * 0.9;

    float y1 = rect.y + rect.h * 0.5;

    glUseProgram(m_widgetProgram);

    glUniform4fv( m_fillColorUniform, 1, s_colors[fillColorId]);
    glUniform4fv( m_borderColorUniform, 1, s_colors[borderColorId]);
    glUniform2f( m_zonesUniform, (xb) - 1, (xb) - 2);

    printf("draw minus fill %d border %d, zone %f %f\n", fillColorId, borderColorId, (xb) - 1, (xb) - 2);
    
    glBegin(GL_TRIANGLE_STRIP);
    glTexCoord3f(-xb, -yb, 0);
    glVertex2f( x0, y1 + yoff);
    glTexCoord3f(xb, -yb, 0);
    glVertex2f( x0, y1 - yoff);

    glTexCoord3f(-xb, 0, 0);
    glVertex2f( x0 + xoff , y1 + yoff);
    glTexCoord3f(xb, 0, 0);
    glVertex2f( x0 + xoff, y1 - yoff);

    glTexCoord3f(-xb, 0, 0);
    glVertex2f( x1 - xoff , y1 + yoff);
    glTexCoord3f(xb, 0, 0);
    glVertex2f( x1 - xoff, y1 - yoff);

    glTexCoord3f(-xb, -yb, 0);
    glVertex2f( x1, y1 + yoff);
    glTexCoord3f(xb, -yb, 0);
    glVertex2f( x1, y1 - yoff);
    glEnd();

    glUseProgram(0);
}

void drawPlus( const nv::Rect& rect, int width, int fillColorId, int borderColorId ) 
{
    float xb = width;
    float yb = width;

    float xoff = xb ;
    float yoff = yb ;

    float x0 = rect.x + rect.w * 0.1 ;
    float x1 = rect.x + rect.w * 0.5;
    float x2 = rect.x + rect.w * 0.9;

    float y0 = rect.y + rect.h * 0.1;
    float y1 = rect.y + rect.h * 0.5;
    float y2 = rect.y + rect.h * 0.9;

    glUseProgram(m_widgetProgram);

    glUniform4fv( m_fillColorUniform, 1, s_colors[fillColorId]);
    glUniform4fv( m_borderColorUniform, 1, s_colors[borderColorId]);
    glUniform2f( m_zonesUniform, (xb) - 1, (xb) - 2);

    printf("draw plus fill %d border %d, zone %f %f\n", fillColorId, borderColorId, (xb) - 1, (xb) - 2);

    glBegin(GL_TRIANGLE_STRIP);
    glTexCoord3f(-xb, -yb, 0);
    glVertex2f( x0, y1 + yoff);
    glTexCoord3f(xb, -yb, 0);
    glVertex2f( x0, y1 - yoff);

    glTexCoord3f(-xb, 0, 0);
    glVertex2f( x0 + xoff , y1 + yoff);
    glTexCoord3f(xb, 0, 0);
    glVertex2f( x0 + xoff, y1 - yoff);

    glTexCoord3f(-xb, 0, 0);
    glVertex2f( x2 - xoff , y1 + yoff);
    glTexCoord3f(xb, 0, 0);
    glVertex2f( x2 - xoff, y1 - yoff);

    glTexCoord3f(-xb, -yb, 0);
    glVertex2f( x2, y1 + yoff);
    glTexCoord3f(xb, -yb, 0);
    glVertex2f( x2, y1 - yoff);
    glEnd();

    glBegin(GL_TRIANGLE_STRIP);
    glTexCoord3f(-xb, -yb, 0);
    glVertex2f( x1 + yoff, y0);
    glTexCoord3f(xb, -yb, 0);
    glVertex2f( x1 - yoff, y0);

    glTexCoord3f(-xb, 0, 0);
    glVertex2f( x1 + yoff, y0 + yoff);
    glTexCoord3f(xb, 0, 0);
    glVertex2f( x1 - yoff, y0 + yoff);

    glTexCoord3f(-xb, 0, 0);
    glVertex2f( x1 + yoff, y2 - yoff);
    glTexCoord3f(xb, 0, 0);
    glVertex2f( x1 - yoff, y2 - yoff);

    glTexCoord3f(-xb, -yb, 0);
    glVertex2f( x1 + yoff, y2);
    glTexCoord3f(xb, -yb, 0);
    glVertex2f( x1 - yoff, y2);
    glEnd();
    /**/
    glUseProgram(0);
}

void drawDownArrow( const nv::Rect& rect, int width, int fillColorId, int borderColorId ) 
{
    float offset = sqrt(2.0) / 2.0 ;

    float xb = width;
    float yb = width;

    float xoff = offset * xb ;
    float yoff = offset * yb ;
    float xoff2 = offset * xb * 2.0 ;
    float yoff2 = offset * yb * 2.0;

    float x0 = rect.x + xoff2;
    float x1 = rect.x + rect.w * 0.5;
    float x2 = rect.x + rect.w - xoff2;

    float y0 = rect.y + rect.h * 0.1 + yoff2;
    float y1 = rect.y + rect.h * 0.6;

    glUseProgram(m_widgetProgram);

    glUniform4fv( m_fillColorUniform, 1, s_colors[fillColorId]);
    glUniform4fv( m_borderColorUniform, 1, s_colors[borderColorId]);
    glUniform2f( m_zonesUniform, (xb) - 1, (xb) - 2);

    glBegin(GL_TRIANGLE_STRIP);
    glTexCoord3f(-xb, -yb, 0);
    glVertex2f( x0, y1 + yoff2);
    glTexCoord3f(xb, -yb, 0);
    glVertex2f( x0 - xoff2, y1);

    glTexCoord3f(-xb, 0, 0);
    glVertex2f( x0 + xoff, y1 + yoff);
    glTexCoord3f(xb, 0, 0);
    glVertex2f( x0 - xoff, y1 - yoff);

    glTexCoord3f(-xb, 0, xb);
    glVertex2f( x1, y0 + yoff2);
    glTexCoord3f(xb, 0, xb);
    glVertex2f( x1 - xoff2, y0);

    glTexCoord3f(xb, 2*yb, xb);
    glVertex2f( x1, y0 - yoff2);

    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glTexCoord3f(xb, -yb, 0);
    glVertex2f( x2 + xoff2, y1);
    glTexCoord3f(-xb, -yb, 0);
    glVertex2f( x2, y1 + yoff2);

    glTexCoord3f(xb, 0, xb);
    glVertex2f( x2 + xoff, y1 - yoff);
    glTexCoord3f(-xb, 0, xb);
    glVertex2f( x2 - xoff, y1 + yoff);

    glTexCoord3f(xb, 0, xb);
    glVertex2f( x1 + xoff2, y0);
    glTexCoord3f(-xb, 0, xb);
    glVertex2f( x1, y0 + yoff2);

    glTexCoord3f(xb, 2*yb, xb);
    glVertex2f( x1, y0 - yoff2);

    glEnd();

    glUseProgram(0);
}

void drawUpArrow( const nv::Rect& rect, int width, int fillColorId, int borderColorId ) 
{
    float offset = sqrt(2.0) / 2.0 ;

    float xb = width;
    float yb = width;

    float xoff = offset * xb ;
    float yoff = - offset * yb ;
    float xoff2 = offset * xb * 2.0 ;
    float yoff2 = - offset * yb * 2.0;

    float x0 = rect.x + xoff2;
    float x1 = rect.x + rect.w * 0.5;
    float x2 = rect.x + rect.w - xoff2;

    float y0 = rect.y + rect.h * 0.9 + yoff2;
    float y1 = rect.y + rect.h * 0.4;

    glUseProgram(m_widgetProgram);

    glUniform4fv( m_fillColorUniform, 1, s_colors[fillColorId]);
    glUniform4fv( m_borderColorUniform, 1, s_colors[borderColorId]);
    glUniform2f( m_zonesUniform, (xb) - 1, (xb) - 2);

    glBegin(GL_TRIANGLE_STRIP);
    glTexCoord3f(-xb, -yb, 0);
    glVertex2f( x0, y1 + yoff2);
    glTexCoord3f(xb, -yb, 0);
    glVertex2f( x0 - xoff2, y1);

    glTexCoord3f(-xb, 0, 0);
    glVertex2f( x0 + xoff, y1 + yoff);
    glTexCoord3f(xb, 0, 0);
    glVertex2f( x0 - xoff, y1 - yoff);

    glTexCoord3f(-xb, 0, xb);
    glVertex2f( x1, y0 + yoff2);
    glTexCoord3f(xb, 0, xb);
    glVertex2f( x1 - xoff2, y0);

    glTexCoord3f(xb, 2*yb, xb);
    glVertex2f( x1, y0 - yoff2);

    glEnd();
    glBegin(GL_TRIANGLE_STRIP);
    glTexCoord3f(xb, -yb, 0);
    glVertex2f( x2 + xoff2, y1);
    glTexCoord3f(-xb, -yb, 0);
    glVertex2f( x2, y1 + yoff2);

    glTexCoord3f(xb, 0, xb);
    glVertex2f( x2 + xoff, y1 - yoff);
    glTexCoord3f(-xb, 0, xb);
    glVertex2f( x2 - xoff, y1 + yoff);

    glTexCoord3f(xb, 0, xb);
    glVertex2f( x1 + xoff2, y0);
    glTexCoord3f(-xb, 0, xb);
    glVertex2f( x1, y0 + yoff2);

    glTexCoord3f(xb, 2*yb, xb);
    glVertex2f( x1, y0 - yoff2);

    glEnd();

    glUseProgram(0);
}

void drawText( const nv::Rect& r, const char * text, int nbLines, int caretPos, bool isHover, bool isOn, bool /*isFocus*/ ) {
    if (isHover || isOn /* || isFocus*/) {
        drawRect(r, cFontBack + (isHover) + (isOn << 1), cOutline);
    }

    glColor4fv(s_colors[cFont]);
    drawString(r.x, r.y, text, nbLines);

    if (caretPos != -1) {
        int w = getTextLineWidthAt( text, caretPos);

        drawRect(nv::Rect( r.x + w, r.y, 2, r.h), cOutline, cOutline);
    }
}

void drawFrame( const nv::Rect& rect, const nv::Point& corner, bool isHover, bool isOn, bool /*isFocus*/ ) 
{
    int lColorNb = cBase + (isHover) + (isOn << 1);// + (isFocus << 2);

    if (corner.x + corner.y == 0)
        drawRect( rect , lColorNb, cOutline);
    else
        drawRoundedRect( rect, corner , lColorNb, cOutline );
}

void drawBoolFrame( const nv::Rect& rect, const nv::Point& corner, bool isHover, bool isOn, bool /*isFocus*/ ) 
{
    int lColorNb = cBool + (isHover) + (isOn << 1);// + (isFocus << 2);

    drawRoundedRect( rect, corner , lColorNb, cOutline );
}

void drawDebugRect(const nv::Rect & rect) {
    glBegin(GL_LINE_STRIP);
    glVertex2i( rect.x + 1, rect.y + 1);
    glVertex2i( rect.x + rect.w, rect.y + 1);
    glVertex2i( rect.x + rect.w, rect.y + rect.h);
    glVertex2i( rect.x + 1, rect.y + rect.h);
    glVertex2i( rect.x + 1, rect.y);
    glEnd();
}
#endif

void draw( )
{
    drawRect(nv::Rect(10, 10, 100, 100), 0, 1);
    drawRect(nv::Rect(110, 10, 100, 100), 1, 1);
    //~ drawRoundedRect(nv::Rect(220, 20, 100, 100), nv::Point(10, 10), 0, 1);
}


int main( )
{
    // create opengl context
    SDL_GL_SetAttribute(SDL_GL_CONTEXT_MAJOR_VERSION, 3);
    SDL_GL_SetAttribute(SDL_GL_CONTEXT_MINOR_VERSION, 3);
    
    if(context_flags != 0)
        SDL_GL_SetAttribute(SDL_GL_CONTEXT_FLAGS, context_flags);
    if(profile_flags != 0)
        SDL_GL_SetAttribute(SDL_GL_CONTEXT_PROFILE_MASK, profile_flags);

    window= SDL_CreateWindow("gKit", 
        SDL_WINDOWPOS_UNDEFINED, SDL_WINDOWPOS_UNDEFINED,
        width, height, SDL_WINDOW_OPENGL);
    if(window == NULL)
        return 1;
    
    gl_context= SDL_GL_CreateContext(window);
    if(gl_context == NULL)
        return 1;
    
    // import opengl extensions
    glewExperimental= 1;        // work around a glew bug (core profile)
    GLenum err= glewInit();
    if(err != GLEW_OK)
    {
        printf("%s\n", glewGetErrorString(err));
        return 1;
    }
    
    while(glGetError() != GL_NO_ERROR) 
        {;}     // workaround a glew bug 

    // display debug messages when possible
    if(context_flags & SDL_GL_CONTEXT_DEBUG_FLAG)
    {
        glDebugMessageCallbackARB(debug, NULL);
        glEnable(GL_DEBUG_OUTPUT_SYNCHRONOUS_ARB);
    }

    // run
    int stop= init();
    if(stop < 0)
        printf("init failed.\n");
    
    while(stop == 0)
    {
        SDL_Event event;
        if(SDL_PollEvent(&event))
        {
            switch(event.type)
            {
                case SDL_WINDOWEVENT_RESIZED:
                    width= event.window.data1;
                    height= event.window.data2;
                    SDL_SetWindowSize(window, width, height);
                    break;
                    
                case SDL_QUIT:
                    stop= 1;
                    break;
            }
        }
        
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        glViewport(0, 0, width, height);
        widgets.reshape( nv::Rect(0, 0, width, height) );
        
        draw();
        
        widgets.draw();
        widgets.clear();
        SDL_GL_SwapWindow(window);
    }
    
    // close
    SDL_GL_DeleteContext(gl_context);
    SDL_DestroyWindow(window);
    SDL_Quit();
    return 0;
}
