
#ifndef _GKSHADER_MANAGER_H
#define _GKSHADER_MANAGER_H

#include "GLManager.h"
#include "GL/GLShaderObjectIO.h"
#include "GL/TPShaderProgram.h"


namespace gk {

//! \ingroup openGL
//@{

//! gestion 'auto' des ressources openGL : pour les shader programs opengl > 2.
inline
GLShaderProgram *createShaderProgram( 
    const TextFile *vertex, 
    const TextFile *fragment,
    const std::vector<std::string> *definitions= NULL )
{
    GLShaderProgram *program= GLManager<GLShaderProgram>::manager().insert( 
        new GLShaderProgram());

    if(vertex != NULL)
    {
        GLShaderObject *shader= GLManager<GLShaderObject>::manager().insert( 
            new GLVertexShader());
        shader->pushSource(vertex);
        if(definitions != NULL)
            shader->setDefinitions(*definitions);
        program->attachShader(shader);
    }
    if(fragment != NULL)
    {
        GLShaderObject *shader= GLManager<GLShaderObject>::manager().insert( 
            new GLFragmentShader());
        shader->pushSource(fragment);
        if(definitions != NULL)
            shader->setDefinitions(*definitions);
        program->attachShader(shader);
    }
    
    return program;
}

//! recharge les sources et reconfigure tous les shaders d'un shader program cree par createShaderProgram( ).
inline
int reloadShaderProgram( GLShaderProgram *program )
{
    if(program == NULL)
        return -1;
    
    GLShaderObject *shaders[GLShaderObject::SHADERTYPE_LAST];
    for(unsigned int i= 0; i < GLShaderObject::SHADERTYPE_LAST; i++)
    {
        GLShaderObject *shader= program->shader(i);
        shaders[i]= shader;     // conserve le shader
        if(shader == NULL)
            continue;
        
        // recupere la configuration du shader
        // cas particulier, les TextFile dupliquent les donnees, pas de recherche dans le manager avec l'objet lui meme.
        TextFile *source= TextFileIO::reload(shader->source()->name());
        if(source == NULL)
            return -1;
        
        std::vector<std::string> definitions;
        if(shader->definitions() != NULL)
            definitions= *shader->definitions();        // duplique les definitions 
        
        // reconfigure le shader
        shader->clear();
        shader->pushSource(source);
        shader->setDefinitions(definitions);
    }
    
    // reconfigure le shader program
    program->clear();
    for(unsigned int i= 0; i < GLShaderObject::SHADERTYPE_LAST; i++)
        if(shaders[i] != NULL)
            program->attachShader(shaders[i]);
    
    if(program->createGLResource() < 0)
        return -1;

    return 1;
}

//! gestion 'auto' des ressources openGL : pour les shader programs opengl > 2.
inline
GLShaderProgram *createShaderProgram( 
    const std::string& vertex, 
    const std::string& fragment, 
    const std::vector<std::string> *definitions= NULL )
{
    return createShaderProgram(
        TextFileIO::read(vertex),
        TextFileIO::read(fragment), 
        definitions);
}

#if defined GK_OPENGL3 || defined GK_OPENGL4    
//! gestion 'auto' des ressources openGL : pour les shader programs opengl > 3.
inline
GLShaderProgram *createShaderProgram( 
    const TextFile *vertex, 
    const TextFile *geometry,
    const TextFile *fragment,
    const std::vector<std::string> *definitions= NULL )
{
    GLShaderProgram *program= createShaderProgram(vertex, fragment, definitions);
    if(program == NULL)
        return NULL;

    if(geometry != NULL)
    {
        GLShaderObject *shader= GLManager<GLShaderObject>::manager().insert( 
            new GLGeometryShader());
        shader->pushSource(geometry);
        if(definitions != NULL)
            shader->setDefinitions(*definitions);
        program->attachShader(shader);
    }

    return program;
}

//! gestion 'auto' des ressources openGL : pour les shader programs opengl > 3.
inline
GLShaderProgram *createShaderProgram( 
    const std::string& vertex, 
    const std::string& geometry,
    const std::string& fragment,
    const std::vector<std::string> *definitions= NULL )    
{
    return createShaderProgram(
        TextFileIO::read(vertex),
        TextFileIO::read(geometry),
        TextFileIO::read(fragment),
        definitions);
}
#endif

#ifdef GK_OPENGL4    
//! gestion 'auto' des ressources openGL : pour les shader programs opengl > 4.
inline
GLShaderProgram *createShaderProgram( 
    const TextFile *vertex, 
    const TextFile *control,
    const TextFile *evaluation,
    const TextFile *geometry,
    const TextFile *fragment,
    const std::vector<std::string> *definitions= NULL )
{
    GLShaderProgram *program= createShaderProgram(vertex, geometry, fragment, definitions);
    if(program == NULL)
        return NULL;

    if(control != NULL)
    {
        GLShaderObject *shader= GLManager<GLShaderObject>::manager().insert( 
            new GLControlShader());
        shader->pushSource(control);
        if(definitions != NULL)
            shader->setDefinitions(*definitions);
        program->attachShader(shader);
    }
    if(evaluation != NULL)
    {
        GLShaderObject *shader= GLManager<GLShaderObject>::manager().insert( 
            new GLEvaluationShader());
        shader->pushSource(evaluation);
        if(definitions != NULL)
            shader->setDefinitions(*definitions);
        program->attachShader(shader);
    }
    
    return program;
}

//! gestion 'auto' des ressources openGL : pour les shader programs opengl > 4.
inline
GLShaderProgram *createShaderProgram( 
    const std::string& vertex, 
    const std::string& control,
    const std::string& evaluation,
    const std::string& geometry,
    const std::string& fragment,
    const std::vector<std::string> *definitions= NULL )
{
    return createShaderProgram(
        TextFileIO::read(vertex),
        TextFileIO::read(control),
        TextFileIO::read(evaluation),
        TextFileIO::read(geometry),
        TextFileIO::read(fragment),
        definitions);
}
#endif

//@}

}

#endif
