
#ifndef _IOMANAGER_H
#define _IOMANAGER_H

#include <string>
#include <map>

//~ #define VERBOSE_DEBUG 1

#include "IOResource.h"

#ifdef VERBOSE_DEBUG
#include <cstdio>
#include <typeinfo>

#ifdef __GLIBCXX__
#include <cxxabi.h>
#include <cstdlib>
#include <cstring>

//! decode les symboles g++ et libere la chaine de caracteres allouee par abi::__cxa_demangle
class demangle
{
    char *name;
    
public:
    demangle( const char *type_name )
        :
        name(NULL)
    {
        int status;
        name= abi::__cxa_demangle(type_name, NULL, NULL, &status);
        if(status != 0)
            name= strdup(type_name);
    }

    ~demangle( )
    {
        if(name != NULL)
            free(name);
    }
    
    operator const char *( ) const
    {
        return name;
    }
    
    operator char *( )
    {
        return name;
    }
};
#else
//! meme interface pour decoder les symboles g++ sur les autres plateformes. mais ne fait rien.
class demangle
{
    const char *name;
    
public:
    demangle( const char *type_name )
        :
        name(type_name)
    {}
    
    ~demangle( ) {}
    
    operator const char *( ) const
    {
        return name;
    }
    
    operator const char *( ) 
    {
        return name;
    }
};
#endif
#endif  // verbose_debug / demangle


namespace gk {

namespace io {

struct object_name 
{
    std::string filename;
    std::string name;
    
    object_name( const std::string& _file, const std::string& _name )
        :
        filename(_file),
        name(_name)
    {}
    
    bool operator< ( const object_name& b ) const
    {
        int cmp= filename.compare(b.filename);
        if(cmp < 0)
            return true;
        else if(cmp > 0)
            return false;
        else
            return (name < b.name);
    }
};

template< typename Object >
struct object_info
{
    IOName ioname;
    Object *object;
    
    object_info( const IOName& _name, Object *_object )
        :
        ioname(_name),
        object(_object)
    {}
};

}       // namespace io


//! manager pour les 'objets' importes a partir d'un fichier.
//! object doit deriver de IOResource.
template< class Object >
class IOManager
{
    // non copyable
    IOManager( const IOManager& );
    IOManager& operator=( const IOManager& );

protected:
    //! constructeur.
    IOManager( )
        :
        m_names_map(),
        m_objects_map(),
        m_objects()
    {}
    
    //! destructeur.
    virtual ~IOManager( )
    {
        int n= (int) m_objects.size();
        for(int i= 0; i < n; i++)
            delete m_objects[i].object;
    }

    typedef io::object_name object_name;
    typedef std::map<object_name, int> names_map_type;
    names_map_type m_names_map;

    typedef std::map<Object *, int> objects_map_type;
    objects_map_type m_objects_map;
    
    typedef io::object_info<Object> object_info;
    std::vector< object_info > m_objects;
    
    object_info *find_info( const std::string& filename, const std::string& name )
    {
        typename names_map_type::iterator 
            found= m_names_map.find( object_name(filename, name) );
        
        if(found == m_names_map.end())
            return NULL;
        else
            return &m_objects[found->second];
    }
    
    object_info *find_object( Object *object )
    {
        typename objects_map_type::iterator 
            found= m_objects_map.find( object );
        
        if(found == m_objects_map.end())
            return NULL;
        else
            return &m_objects[found->second];
    }
    
public:
    //! reference un nouvel 'objet' par son nom et le nom du fichier d'import.
    Object *insert( Object *object, const IOFileInfo& handle, const std::string& name= "" )
    {
        if(object == NULL)
            return NULL;
        
    #ifdef VERBOSE_DEBUG
        printf("IOManager<%s> %p::insert( ): object %p '%s', filename '%s'\n", 
            (const char *) demangle(typeid(Object).name()), this,
            object, name.c_str(), handle.file->filename.c_str());
    #endif
        
        m_names_map.insert( std::make_pair( object_name(handle.filename(), name), m_objects.size() ));
        m_objects_map.insert( std::make_pair( object, m_objects.size() ));
        m_objects.push_back( object_info( IOName(handle, name), object ));
        return object;
    }

    //! \todo temporaire le temps de porter les autres IOMachins
    Object *insert( Object *object, const std::string& filename, const std::string& name= "" )
    {
    #ifdef VERBOSE_DEBUG
        printf("IOManager<%s> %p::insert( ): object %p '%s', filename '%s'\n", 
            (const char *) demangle(typeid(Object).name()), this,
            object, name.c_str(), filename.c_str());
    #endif
        
        IOFileInfo handle= IOFileManager::manager().file(filename);
        if(handle.isValid() == false)
            return NULL;
        
        return insert(object, handle, name);
    }
    
    //! recherche un 'objet' deja importe.
    Object *find( const IOFileInfo& handle, const std::string& name= "" )
    {
        object_info *found= find_info(handle.filename(), name);
    #ifdef VERBOSE_DEBUG
        printf("IOManager<%s> %p::find( ): object '%s', filename '%s'... ", 
            (const char *) demangle(typeid(Object).name()), this,
            name.c_str(), handle.filename().c_str());
        if(found != NULL)
            printf("found object %p done.\n", found->object);
        else
            printf("failed.\n");
    #endif

        if(found == NULL)
            return NULL;
        return found->object;
    }
    
    //! recherche un objet deja importe.
    Object *find( const std::string& filename, const std::string& name= "" )
    {
        object_info *found= find_info(filename, name);
    #ifdef VERBOSE_DEBUG
        printf("IOManager<%s> %p::find( ): object '%s', filename '%s'... ", 
            (const char *) demangle(typeid(Object).name()), this,
            name.c_str(), filename.c_str());
        if(found != NULL)
            printf("found object %p done.\n", found->object);
        else
            printf("failed.\n");
    #endif

        if(found == NULL)
            return NULL;
        return found->object;
    }
    
    //! recherche le nom et le nom du fichier d'un 'objet' deja importe.
    IOName *find( Object *object )
    {
        object_info *found= find_object(object);
    #ifdef VERBOSE_DEBUG
        printf("IOManager<%s> %p::find( ): object %p... ", 
            (const char *) demangle(typeid(Object).name()), this, object);
        
        if(found != NULL)
            printf("found object '%s', filename '%s' done.\n", 
                found->ioname.name().c_str(), found->ioname.filename().c_str());
        else
            printf("failed.\n");
    #endif
        
        if(found == NULL)
            return NULL;
        return &found->ioname;
    }
    
    //! renvoie l'etat du fichier associe a un objet gere par le manager.
    //! -1 en cas d'erreur, 0 pas de modifications, 1 le fichier a ete modifie depuis sa lecture par le manager.
    int modified( Object *object )
    {
        object_info *found= find_object(object);
        if(found == NULL || found->ioname->handle().isValid() == false)
        {
            // le fichier n'est pas reference ... pas lu en passant par le manager.
            printf("modified %p... failed.\n", object);
            return -1;
        }
        
        // verifie la date et la taille du fichier
        return found->ioname->handle().modified();
    }
    
    //! renvoie l'etat d'un fichier charge par le manager.
    //! -1 en cas d'erreur, 0 pas de modifications, 1 le fichier a ete modifie depuis sa lecture par le manager.
    int modified( const std::string& filename, const std::string& name= "" )
    {
        object_info *found= find_info(filename, name);
        if(found == NULL || found->ioname->handle().isValid() == false)
        {
            // le fichier n'est pas reference ... pas lu en passant par le manager.
            printf("modified '%s'... failed.\n", filename.c_str());
            return -1;
        }
        
        // verifie la date et la taille du fichier
        return found->ioname->handle().modified();
    }
    
    //~ //! detruit un objet deja charge.
    //~ int release( Object *object )
    //~ {
        //~ delete object;
        //~ return 0;
    //~ }
};

//~ #undef VERBOSE_DEBUG
}       // namespace

#endif
