
#ifndef _IOFILE_H
#define _IOFILE_H

#include <cassert>
#include "IOFileSystem.h"


namespace gk
{
    
class IOFileInfo;
class IOFileManager;
    
//! representation interne de l'etat (taille + date) d'un fichier charge par l'application.
//! IOFileManager conserve la description de tous les fichiers charges par l'application.
class IOFile
{
    // non copyable
    IOFile( );
    IOFile( const IOFile& );
    IOFile& operator=( const IOFile& );
    
public:
    std::string filename;       //!< nom du fichier.
    IOInfo info;        //!< taille + date fichier.
    
    IOFile( const std::string& _filename, const IOInfo& _info= IOInfo() )
        :
        filename(_filename),
        info(_info)
    {}
};


//! descripteur de fichier charge par l'application.
//! (representation publique).
class IOFileInfo
{
public:
    IOFile *file;       //!< identifiant du fichier.
    IOInfo info;        //!< version du fichier lu par l'application.
    
    //! constructeur par defaut.
    IOFileInfo( )
        :
        file(NULL),
        info()
    {}
    
    //! constructeur.
    IOFileInfo( IOFile *_file, const IOInfo& _info )
        :
        file(_file),
        info(_info)
    {}
    
    //! constructeur.
    IOFileInfo( IOFile *_file )
        :
        file(_file),
        info()
    {
        if(file != NULL)
            info= _file->info;
    }
    
    ~IOFileInfo( ) {}
    
    IOFileInfo( const IOFileInfo& b ) 
        :
        file(b.file),
        info(b.info)
    {}
    
    IOFileInfo& operator=( const IOFileInfo & b )
    {
        file= b.file;
        info= b.info;
        return *this;
    }
    
    //! renvoie vrai si le descripteur de fichier est correct / initialise.
    bool isValid( ) const
    {
        return (file != NULL);
    }
    
    //! renvoie 1 si le fichier a ete modifie depuis son chargement par l'application.
    int modified( ) const
    {
        if(file == NULL)
            return -1;
        
        // compare la date des infos a celles du fichier
        return (info != file->info) ? 1 : 0;
    }
    
    //! met a jour le descripteur.
    int update( )
    {
        if(file ==  NULL)
            return -1;
        
        info= file->info;
        return 0;
    }

    //! renvoie le nom du fichier designe par le descripteur.
    const std::string& filename( ) const
    {
        assert(file != NULL);
        return file->filename;
    }
};

}       // namespace gk

#endif
