
#ifndef _TP_SHADER_PROGRAM_H
#define _TP_SHADER_PROGRAM_H

#include <string>
#include <vector>

#include "GLManager.h"
#include "GL/GLPlatform.h"
#include "GLResource.h"
#include "GL/GLShaderObject.h"
#include "GL/TPProgramName.h"


namespace gk {

//! representation d'un shader program openGL.
class GLShaderProgram : public GLResource
{
    struct parameter
    {
        std::string name;
        int location;
        int index;
        int size;
        GLenum type;
        bool is_integer;
        
        parameter( )
            :
            name(),
            location(-1),
            index(-1),
            size(0),
            type(0),
            is_integer(false)
        {}
        
        parameter( const char *_name, const int _location, const int _index, const int _size, const GLenum _type, const bool _is_integer= false )
            :
            name(_name),
            location(_location),
            index(_index),
            size(_size),
            type(_type),
            is_integer(_is_integer)
        {}
        
        ~parameter( ) {}
    };
    
private:
    // non copyable
    GLShaderProgram( const GLShaderProgram& );
    GLShaderProgram& operator=( const GLShaderProgram& );
    
protected:
    GLShaderObject *m_shaders[GLShaderObject::SHADERTYPE_LAST];
    
    std::vector<parameter> m_feedbacks;
    std::vector<parameter> m_attributes;
    std::vector<parameter> m_uniforms;
    std::vector<parameter> m_samplers;

    int m_attribute_count;
    int m_uniform_count;
    int m_sampler_count;
    int m_feedback_count;

    bool m_is_linked;
    bool m_is_validated;

    int make( );

public:
    static bool is_sampler( const GLenum type );
    static bool is_integer( const GLenum type );
    
    //! constructeur par defaut.
    GLShaderProgram( );

    //! desctructeur.
    virtual ~GLShaderProgram( ) {}

    //! ajoute un shader object au shader program.
    int attachShader( GLShaderObject *shader );

    //! renvoie un shader attache au shader program, cf gk::GLShaderObject::VERTEX, etc.
    GLShaderObject *shader( const unsigned int type );
    
    //! (re-)linke le shader program.
    int link( );
    //! valide la configuration du shader program.
    int validate( );

    //! construit le shader program. creation de l'objet openGL.
    int createGLResource( );

    //! detruit l'objet openGL.
    int releaseGLResource( );

    //! reinitialise l'etat du shader program.
    int clear( );
    
    //! renvoie le nombre d'uniforms.
    int uniformCount( ) const
    {
        return m_uniform_count;
    }
    
    const char *uniformName( const ProgramUniform& uniform ) const
    {
        return m_uniforms[uniform.index()].name.c_str();
    }
    
    //! renvoie l'identifiant d'un uniform du shader program.
    ProgramUniform uniform( const char *name ) const;

    //! renvoie le nombre de samplers.
    int samplerCount( ) const
    {
        return m_sampler_count;
    }

    const char *samplerName( const ProgramSampler& sampler ) const
    {
        return m_samplers[sampler.index()].name.c_str();
    }
    
    //! renvoie l'identifiant d'un sampler du shader program.
    ProgramSampler sampler( const char *name ) const;
    
    //! renvoie l'identifiant d'un uniform block du shader program.
    ProgramInterface interface( const char *name ) const;
    
    //! renvoie le nombre d'attributs.
    int attributeCount( ) const
    {
        return m_attribute_count;
    }
    
    //! renvoie l'identifiant d'un attribut du shader program.
    ProgramAttribute attribute( const char *name ) const;

    const char *attributeName( const ProgramAttribute& attribute ) const
    {
        return m_attributes[attribute.index()].name.c_str();
    }
    
    //~ //! renvoie le nombre de drawbuffer (varying out du fragment shader).
    //~ int drawbufferCount( ) const
    //~ {
        //~ //! \todo
        //~ return -1;
    //~ }
    
    //! renvoie le draw buffer d'un varying du fragment shader.
    ProgramDrawbuffer drawbuffer( const char *name ) const ;
    
    //! renvoie le nombre de feedbacks
    int feedbackCount( ) const
    {
        return m_feedback_count;
    }
    
    //! renvoie l'identifiant du buffer d'un varying utilise par le transform feedback.
    ProgramFeedback feedback( const char *name ) const;
};

}

#endif
