
#ifndef _GLSHADER_OBJECT_H
#define _GLSHADER_OBJECT_H

#include <vector>
#include <string>

#include "GL/GLPlatform.h"
#include "GLResource.h"
#include "TextFile.h"


namespace gk {

//! utilisation interne. abstraction d'un shader object, ressource openGL.
class GLShaderObject : public GLResource
{
private:
    // non copyable
    GLShaderObject( );
    GLShaderObject( const GLShaderObject& );
    GLShaderObject& operator= ( const GLShaderObject& );

protected:
    TextFile *m_text;

    unsigned int m_type;
    bool m_errors;

public:
    //! types de shaders.
    enum
    {
        VERTEX= 0,      //!< type d'un vertex shader,
        FRAGMENT,       //!< type d'un fragment shader,
        GEOMETRY,       //!< type d'un geometry shader,
        CONTROL,        //!< type d'un control shader,
        EVALUATION,     //!< type d'un evaluation shader.
        SHADERTYPE_LAST
    };
    
    static
    GLenum ShaderType[];
    
    static
    const char *ShaderTypeString[];
    
    //! constructeur, 
    //! \param type correspond a une contante gk::GLShaderObject::VERTEX, etc.
    GLShaderObject( const unsigned int type );
    
    //! destructeur.
    virtual ~GLShaderObject( )
    {
        delete m_text;
    }

    //! ajoute une definition au source. "#define 'definition' 'value'".
    int pushDefinition( const std::string& definition, const std::string& value= "" )
    {
        if(m_text == NULL)
            return -1;
        return m_text->define(definition, value);
    }

    //! remplace l'ensemble de definitions.
    int setDefinitions( const std::vector<std::string>& definitions )
    {
        if(m_text == NULL)
            return -1;
        return m_text->setDefinitions(definitions);
    }
    
    //! renvoie les definitions.
    const std::vector<std::string> *definitions( )
    {
        if(m_text == NULL)
            return NULL;
        return m_text->definitions();
    }
    
    //! ajoute le contenu d'un fichier source.
    int pushSource( const TextFile *text )
    {
        if(text == NULL)
            return 0;
        
        if(m_text == NULL)
        {
            //~ printf("ShaderObject: source '%s'...\n", text->name().c_str());
            m_text= new TextFile(text->name());
            if(m_text == NULL)
                return -1;
        }
        
        return m_text->include(text);
    }

    //! renvoie le source du shader.
    TextFile *source( ) const
    {
        return m_text;
    }
    
    //! affiche les sources du shader.
    void printSource( ) const
    {
        if(m_text == NULL)
            return;
        m_text->print();
    }

    //! affiche les erreurs de compilation du shader.
    void printErrors( );

    //! cree l'objet openGL.
    int createGLResource( );
    //! detruit l'objet openGL.
    int releaseGLResource( );

    //! reinitialise l'etat du shader.
    int clear( )
    {
        delete m_text;
        m_text= NULL;
        m_errors= false;
        return 0;
    }
    
    //! renvoie vrai ou faux, si erreurs de lecture du source.
    bool errors( ) const
    {
        return m_errors;
    }

    //! renvoie le type du shader.
    unsigned int type( ) const
    {
        return m_type;
    }
};


//! representation d'un vertex shader object, objet openGL.
class GLVertexShader : public GLShaderObject
{
public:
    //! constructeur par defaut.
    GLVertexShader( )
        :
        GLShaderObject(GLShaderObject::VERTEX)
    {}

    //! destructeur.
    ~GLVertexShader( ) {}
};

//! representation d'un fragment shader object, objet openGL.
class GLFragmentShader: public GLShaderObject
{
public:
    //! constructeur par defaut.
    GLFragmentShader( )
        :
        GLShaderObject(GLShaderObject::FRAGMENT)
    {}

    //! destructeur.
    ~GLFragmentShader( ) {}
};


#if defined GK_OPENGL3 || defined GK_OPENGL4
//! representation d'un geometry shader object. objet openGL.
class GLGeometryShader : public GLShaderObject
{
public:
    GLGeometryShader( )
        :
        GLShaderObject(GLShaderObject::GEOMETRY)
    {}

    ~GLGeometryShader( ) {}
};
#endif

#ifdef GK_OPENGL4
//! representation d'un tesselation control shader object. objet openGL.
class GLControlShader : public GLShaderObject
{
public:
    GLControlShader( )
        :
        GLShaderObject(GLShaderObject::CONTROL)
    {}

    ~GLControlShader( ) {}
};

//! representation d'un tesselation evaluation shader object. objet openGL.
class GLEvaluationShader : public GLShaderObject
{
public:
    GLEvaluationShader( )
        :
        GLShaderObject(GLShaderObject::EVALUATION)
    {}

    ~GLEvaluationShader( ) {}
};
#endif

}       // namespace
#endif
