
#ifndef _GKEFFECT_SHADER_MANAGER_H
#define _GKEFFECT_SHADER_MANAGER_H

#include <string>

#include "Name.h"
#include "Effect.h"
#include "IOFileManager.h"
#include "GL/TPShaderProgram.h"


namespace gk {

//! creation simplifiee d'un shader program a partir d'un script.
class  EffectShaderManager 
{
    Effect *m_effect;
    
    std::map<Name, int> m_programs_map;
    std::vector<GLShaderProgram *> m_programs;
    std::vector<std::string> m_program_names;
    
    // non copyable.
    EffectShaderManager( const EffectShaderManager& );
    EffectShaderManager& operator= ( const EffectShaderManager& );
    
public:
    //! constructeur par defaut
    EffectShaderManager( );

    //! charge le script.
    int read( const std::string& effect );

    //! constructeur. charge le script.
    EffectShaderManager( const std::string& effect );

    //! constructeur. reference un script deja charge.
    EffectShaderManager( Effect *effect );
    
    //! destructeur.
    ~EffectShaderManager( );

    //! recharge le fichier de configuration et reinitialise les shader programs deja references/crees par l'application.
    //! renvoie -1 en cas d'erreur, 0 si le fichier n'est pas modifie et 1 si le fichier est modifie + recharge.
    int reload( );

    //! insere un shader program existant dans le manager.
    GLShaderProgram *insert( GLShaderProgram *program, const std::string& name )
    {
        m_programs_map.insert( std::make_pair( name, m_programs.size()) );
        m_programs.push_back( program );
        m_program_names.push_back( name );
        assert(m_programs.size() == m_program_names.size());
        return program;
    }

    //! retrouve un shader program. renvoie eventuellement NULL si le programe n'existe pas.
    GLShaderProgram *program( const Name& name )
    {
        std::map<Name, int>::iterator found= m_programs_map.find(name);
        if(found == m_programs_map.end())
            return NULL;
        else
            return m_programs[found->second];
    }

    //! retouve un shader program. renvoie eventuellement NULL si le programe n'existe pas.
    GLShaderProgram *operator[] ( const Name &name )
    {
        return program(name);
    }
    
    //! cree un shader program opengl a partir d'une section complete.
    GLShaderProgram *createShaderProgram( const std::string& program_key );
};

}       // namespace gk

#endif
