
#ifndef _IOEFFECT_H
#define _IOEFFECT_H

#include "IOManager.h"
#include "Effect.h"


namespace gk {

//! manager pour importer des effets / scripts.
class  EffectIO : public IOManager<Effect>
{
    // non copyable
    EffectIO( const EffectIO& );
    EffectIO& operator=( const EffectIO& );
    
    // private default constructor, singleton
    EffectIO( )
        :
        IOManager<Effect>( )
    {}
    
public:
    //! importe l'objet 'name' du script 'filename'
    static
    Effect *read( const std::string& filename, const std::string& name= "" ) 
    {
        // importer le fichier, si necessaire
        if(filename.empty() == true)
            return NULL;
        
        IOFileInfo handle= IOFileManager::manager().file(filename);
        if(handle.isValid() == false)
            return NULL;
        Effect *file= manager().find(handle, name);
        if(file != NULL)
            return file;

        // importer le fichier
        file= new Effect;
        if(file->read(filename) < 0)
        {
            delete file;
            return NULL;
        }
        
        // reference le script avec le manager
        return manager().insert(file, filename, name);
    }

    //! renvoie -1 en cas d'erreur, 1 lorsque le fichier a ete recharge, et 0 sinon (le fichier n'a pas change).
    static
    int reload( Effect *object )
    {
        if(object == NULL)
            return -1;
        
        IOName *ioname= manager().find(object);
        if(ioname == NULL || ioname->handle().isValid() == false)
        {
            // le fichier n'est pas reference ... pas lu en passant par le manager.
            printf("reloading effect %p... failed.\n", object);
            return 0;
        }
        
        // verifie la date et la taille du fichier
        if(ioname->handle().modified() <= 0)
            // le fichier n'est pas modifie
            return 0;
        
        // recharger le fichier
        object->clear();
        if(object->read(ioname->filename()) < 0)
        {
        #ifdef VERBOSE      //_DEBUG
            printf("reloading '%s'... failed.\n", ioname->filename().c_str());
        #endif
            return -1;
        }
        
        // mettre a jour les infos sur le fichier
        ioname->handle().update();
    #ifdef VERBOSE      //_DEBUG
        printf("reloading '%s'... done.\n", ioname->filename().c_str());
    #endif
        return 1;
    }
    
    static
    Effect *reload( const std::string& filename, const std::string& name= "" ) 
    {
        if(filename.empty() == true)
            return NULL;

        Effect *object= manager().find(filename, name);
        int code= reload(object);
        if(code < 0)
            return NULL;
        return object;
    }
    
    static
    EffectIO& manager( )  //!< singleton
    {
        static EffectIO manager;
        return manager;
    }
};

}

#endif
