
#include <cstdio>
#include <cstring>

#include "CLInit.h"


namespace gk {

cl_platform_id ActivePlatform= 0;
cl_context ActiveContext= 0;
cl_device_id ActiveDevice= 0;
CLQueue *ActiveQueue= NULL;
unsigned int ActiveContextFlags= 0;
std::string ActiveBuildOptions= std::string();
bool Active= false;

    
int CLInit( const unsigned int flags )
{
    cl_uint platform_count;
    cl_int status= clGetPlatformIDs(0, NULL, &platform_count);
    if(status != CL_SUCCESS)
        return -1;
    
    #ifdef VERBOSE
    {
        printf("CLinit( ): %d openCL platforms.\n", platform_count);
    }
    #endif
    
    cl_platform_id *platforms= new cl_platform_id[platform_count];
    if(platforms == NULL)
        return -1;
    status= clGetPlatformIDs(platform_count, platforms, NULL);
    if(status != CL_SUCCESS)
        return -1;
    
    cl_device_id *cpu_devices= NULL;
    cl_uint cpu_count= 0;
    for(unsigned int i= 0; i < platform_count; i++)
    {
        delete [] cpu_devices;
        cpu_devices= NULL;
        
        status= clGetDeviceIDs(platforms[i], CL_DEVICE_TYPE_CPU, 0, NULL, &cpu_count);
        if(status != CL_SUCCESS)
            continue;
        if(cpu_count == 0)
            continue;
        
        printf("CLinit( ): %d cpu devices / platform %d.\n", cpu_count, i);
        #ifdef VERBOSE
        {
            size_t length= 0;
            cl_int status= clGetPlatformInfo(platforms[i], CL_PLATFORM_VERSION, 0, NULL, &length);
            char *version= new char[length];
            status= clGetPlatformInfo(platforms[i], CL_PLATFORM_VERSION, length, version, NULL);
            printf("  CL version: %s\n", version);
            delete [] version;
        }
        #endif
        
        cpu_devices= new cl_device_id[cpu_count];
        if(cpu_devices == NULL)
            continue;
        status= clGetDeviceIDs(platforms[i], CL_DEVICE_TYPE_CPU, cpu_count, cpu_devices, NULL);
        if(status != CL_SUCCESS)
            continue;
        
        if(flags & CPU_BIT)
        {
            ActivePlatform= platforms[i];
            ActiveDevice= cpu_devices[0];
        #ifdef VERBOSE
            printf("CLInit( ): find CPU device.\n");
        #endif
            break;
        }
    }
    
    cl_device_id gl_shared_device= 0;
    cl_device_id *gpu_devices= NULL;
    cl_uint gpu_count= 0;
    for(unsigned int i= 0; i < platform_count; i++)
    {
        delete [] gpu_devices;
        gpu_devices= NULL;
        
        status= clGetDeviceIDs(platforms[i], CL_DEVICE_TYPE_GPU, 0, NULL, &gpu_count);
        if(status != CL_SUCCESS)
            continue;
        if(gpu_count == 0)
            continue;
        
        printf("CLinit( ): %d gpu devices / platform %d.\n", gpu_count, i);
        gpu_devices= new cl_device_id[gpu_count];
        if(gpu_devices == NULL)
            continue;
        status= clGetDeviceIDs(platforms[i], CL_DEVICE_TYPE_GPU, gpu_count, gpu_devices, NULL);
        if(status != CL_SUCCESS)
            continue;
        
        #ifdef VERBOSE
        {
            size_t length= 0;
            cl_int status= clGetPlatformInfo(platforms[i], CL_PLATFORM_VERSION, 0, NULL, &length);
            char *version= new char[length];
            status= clGetPlatformInfo(platforms[i], CL_PLATFORM_VERSION, length, version, NULL);
            printf("  CL version: %s\n", version);
            delete [] version;
        }
        #endif
        
        // contexte openGL / openCL
        {
            char *extensions= NULL;
            for(unsigned int gpu= 0; gpu < gpu_count; gpu++)
            {
                delete [] extensions;
                extensions= NULL;
                
                size_t extension_length= 0;
                status= clGetDeviceInfo(gpu_devices[gpu], CL_DEVICE_EXTENSIONS, 0, NULL, &extension_length);
                if(status != CL_SUCCESS)
                    continue;
                
                extensions= new char[extension_length];
                status= clGetDeviceInfo(gpu_devices[gpu], CL_DEVICE_EXTENSIONS, extension_length, extensions, NULL);
                if(status != CL_SUCCESS)
                    continue;
                
                #ifdef VERBOSE_DEBUG
                {
                    printf("  extensions:\n    ");
                    for(unsigned int e= 0; extensions[e] != 0; e++)
                    {
                        if(extensions[e] == ' ')
                        {
                            printf("\n    ");
                            continue;
                        }
                        printf("%c", extensions[e]);
                    }
                    printf("\n  done.\n");
                }
                #endif
                
            #ifdef APPLE_OSX
                char *ext= strstr(extensions, "cl_APPLE_gl_sharing");
            #else
                char *ext= strstr(extensions, "cl_khr_gl_sharing");
            #endif
                if(ext == NULL)
                    continue;
                
                gl_shared_device= gpu_devices[gpu];
                
                #ifdef VERBOSE
                {
                    printf("CLInit( ): openGL/CL shared device: ");
                    size_t length= 0;
                    status= clGetDeviceInfo(gpu_devices[gpu], CL_DEVICE_NAME, 0, NULL, &length);
                    char *name= new char[length];
                    status= clGetDeviceInfo(gpu_devices[gpu], CL_DEVICE_NAME, length, name, NULL);
                    printf("'%s', gpu %d / platform %d.\n", name, gpu, i);
                    delete [] name;
                }
                #endif
            }
            
            delete [] extensions;
        }
        
        if((flags & GPU_BIT) || (flags & GL_BIT))
        {
        #ifdef VERBOSE
            printf("CLInit( ): find GPU device.\n");
        #endif
            ActivePlatform= platforms[i];
            if(gl_shared_device != NULL && (flags & GL_BIT) != 0)
            {
            #ifdef VERBOSE
                printf("CLInit( ): find openGL device.\n");
            #endif
                ActiveDevice= gl_shared_device;
            }
            else
                ActiveDevice= gpu_devices[0];   // reprendre et simplifier tout ca, utiliser getDeviceFromType(cpu | gpu)
            break;
        }
    }
    
    delete [] platforms;
    delete [] cpu_devices;
    delete [] gpu_devices;
    
    if((flags & CPU_BIT) != 0)
    {
        cl_context_properties props[]= 
        { 
            CL_CONTEXT_PLATFORM, (cl_context_properties) ActivePlatform, 0
        };
        
        ActiveContext= clCreateContext(props, 1, &ActiveDevice, NULL, NULL, &status);
        if(status != CL_SUCCESS)
        {
            printf("CLInit( ): create cpu context error.\n");
            return -1;
        }
    }
    else if(gl_shared_device != NULL && (flags & (GL_BIT | GPU_BIT)) != 0)
    {
    #ifdef APPLE_OSX
        // initialisation du contexte openCL a partir du contexte openGL
        CGLContextObj CGLContext = CGLGetCurrentContext();
        CGLShareGroupObj CGLShareGroup = CGLGetShareGroup(CGLContext);
        cl_context_properties props[] = 
        {
            CL_CONTEXT_PROPERTY_USE_CGL_SHAREGROUP_APPLE, (cl_context_properties) CGLShareGroup, 
            0 
        };
    #elif defined WIN32
        // initialisation du contexte openCL a partir du contexte openGL
        cl_context_properties props[] = 
        {
            CL_CONTEXT_PLATFORM, (cl_context_properties) ActivePlatform, 
            CL_WGL_HDC_KHR, (cl_context_properties) wglGetCurrentDC(), 
            CL_GL_CONTEXT_KHR, (cl_context_properties) wglGetCurrentContext(), 
            0
        };
    #else
        // initialisation du contexte openCL a partir du contexte openGL
        cl_context_properties props[]= 
        { 
            CL_CONTEXT_PLATFORM, (cl_context_properties) ActivePlatform, 
            CL_GLX_DISPLAY_KHR, (cl_context_properties) glXGetCurrentDisplay(),
            CL_GL_CONTEXT_KHR, (cl_context_properties) glXGetCurrentContext(), 
            0
        };
    #endif
        
        // cl / gl sharing device
        // \todo utiliser directement l'extension pour recuperer le device cl associe au contexte opengl courant. virer l'enumeration...
        clGetGLContextInfoKHR_fn getGLContextInfo= (clGetGLContextInfoKHR_fn) clGetExtensionFunctionAddress("clGetGLContextInfoKHR");
        if(getGLContextInfo != NULL)
        {
            cl_device_id gl_device= 0;
            if(getGLContextInfo(props, CL_CURRENT_DEVICE_FOR_GL_CONTEXT_KHR, sizeof(gl_device), &gl_device, NULL) != CL_SUCCESS)
            {
                printf("CLinit( ): get openGL context error.\n");
                return -1;
            }
            
            ActiveDevice= gl_device;
            printf("CLInit( ): using openGL/CL shared device/context.\n");
        }
        
        ActiveContext= clCreateContext(props, 1, &ActiveDevice, NULL, NULL, &status);
        if(status != CL_SUCCESS)
        {
            printf("CLInit( ): create gpu context error.\n");
            return -1;
        }
    }
    
    // cree une file de commandes par defaut
    cl_command_queue_properties qflags= 0;
    if(flags & PROFILE_BIT )
        qflags= CL_QUEUE_PROFILING_ENABLE;
    
    ActiveQueue= new CLQueue( qflags );
    ActiveContextFlags= flags;
    Active= true;
    
    {
        size_t length= 0;
        clGetDeviceInfo(ActiveDevice, CL_DEVICE_NAME, 0, NULL, &length);
        char *name= new char[length];
        clGetDeviceInfo(ActiveDevice, CL_DEVICE_NAME, length, name, NULL);
        printf("\nusing openCL device '%s':\n", name);
        delete [] name;
        
        cl_uint k;
        clGetDeviceInfo(ActiveDevice, CL_DEVICE_MAX_WORK_ITEM_DIMENSIONS, sizeof(k), &k, NULL);
        
        // affiche quelques informations sur le device
        size_t *work_item_sizes= new size_t[k];
        clGetDeviceInfo(ActiveDevice, CL_DEVICE_MAX_WORK_ITEM_SIZES,  sizeof(size_t) * k, work_item_sizes, NULL);
        printf("  max work item sizes %lu %lu %lu\n", work_item_sizes[0], work_item_sizes[1], work_item_sizes[2]);
        delete [] work_item_sizes;
        
        size_t work_group_size;
        clGetDeviceInfo(ActiveDevice, CL_DEVICE_MAX_WORK_GROUP_SIZE, sizeof(work_group_size), &work_group_size, NULL);
        printf("  max work group size %lu\n", work_group_size);
        
        cl_ulong global_size;
        clGetDeviceInfo(ActiveDevice, CL_DEVICE_GLOBAL_MEM_SIZE, sizeof(global_size), &global_size, NULL);
        printf("  global memory %luMB (0x%lx)\n", global_size / 1024 / 1024, global_size);
        
        cl_ulong local_size;
        clGetDeviceInfo(ActiveDevice, CL_DEVICE_LOCAL_MEM_SIZE, sizeof(local_size), &local_size, NULL);
        printf("  local memory %luKB (0x%lx)\n", local_size / 1024, local_size);
        
        cl_ulong constant_size;
        clGetDeviceInfo(ActiveDevice, CL_DEVICE_MAX_CONSTANT_BUFFER_SIZE, sizeof(constant_size), &constant_size, NULL);
        printf("  constant memory %luKB (0x%lx)\n", constant_size / 1024, constant_size);
        printf("done.\n");
    }
    
    if(flags & DEBUG_BIT)
        ActiveBuildOptions.append("-g ");
    
    return 0;
}

void CLQuit( void )
{
    printf("CLQuit( ).\n");
    if(ActiveQueue != NULL)
    {
        ActiveQueue->releaseCLResource();
        delete ActiveQueue;
        ActiveQueue= NULL;
    }
    
    if(ActiveContext != 0)
    {
        clReleaseContext(ActiveContext);
        ActiveContext= 0;
    }
}

int CLBuildOptions( const std::string& options )
{
    if(Active == false)
        return -1;
    
    ActiveBuildOptions.push_back(' ');
    ActiveBuildOptions.append(options);
    
//~ #ifdef VERBOSE
    //~ printf("CLInit( ): active build options:\n%s\n", ActiveBuildOptions.c_str());
//~ #endif
    
    return 0;
}

}       // namespace
