
#include <cstdio>

#include "CLInit.h"
#include "CLError.h"
#include "CLImage.h"
#include "CLManager.h"


namespace gk {

CLImage::CLImage( const cl_mem_flags flags, const unsigned int glname, const unsigned int texture_target, const int level )
    :
    CLResource(),
    m_image(0)
{
    cl_int status;
    m_image= clCreateFromGLTexture2D(ActiveContext, flags, texture_target, level, glname, &status);
    // deprecated dans opencl 1.2 ... utiliser clCreateFromGLTexture pour tous les types de textures, array, 2d, 3d, etc.
    if(status != CL_SUCCESS || m_image == 0)
    {
        printf("error creating CLImage from openGL texture: %s\n", CLError(status));
        m_image= 0;
        return;
    }
    
    // recupere les proprietes de l'image
    cl_image_format format;
    clGetImageInfo(m_image, CL_IMAGE_FORMAT, sizeof(cl_image_format), &format, NULL);
    clGetImageInfo(m_image, CL_IMAGE_ROW_PITCH, sizeof(size_t), &m_row_pitch, NULL);
    clGetImageInfo(m_image, CL_IMAGE_WIDTH, sizeof(size_t), &m_width, NULL);
    clGetImageInfo(m_image, CL_IMAGE_HEIGHT, sizeof(size_t), &m_height, NULL);
#ifdef VERBOSE
    printf("CLImage(GLTexture) %lux%lu image, channel order 0x%x, type 0x%x.\n", 
        m_width, m_height, 
        format.image_channel_order, format.image_channel_data_type);
#endif
}
    
CLImage::CLImage( const cl_mem_flags flags, const cl_image_format *format, const size_t width, const size_t height, void *data, const size_t pitch )
    :
    CLResource(),
    m_image(0),
    m_width(width),
    m_height(height),
    m_row_pitch(pitch)
{
    cl_int status;
    m_image= clCreateImage2D(ActiveContext, flags, format, width, height, pitch, data, &status);
    if(status != CL_SUCCESS)
    {
        m_image= 0;
        return;
    }
}

int CLImage::createCLResource( )
{
    return (m_image != 0) ? 0 : -1;
}

int CLImage::releaseCLResource( )
{
    if(m_image == 0)
        return -1;

    clReleaseMemObject(m_image);
    m_image= 0;
    return 0;
}

int CLImage::update( const ImageOrigin& origin, const ImageRegion& region, const void *data, const cl_bool block, cl_event *wait )
{
    cl_int status= clEnqueueWriteImage(ActiveQueue->queue(), m_image, block, origin, region, 0, 0, data, 0, NULL, wait);
    return (status == CL_SUCCESS) ? 0 : -1;
}

int CLImage::update( const void *data, const cl_bool block, cl_event *wait )
{
    ImageOrigin origin(0, 0);
    ImageRegion region(m_width, m_height);
    return update(origin, region, data, block, wait);
}

int CLImage::update( const size_t x, const size_t y, const size_t w, const size_t h, const void *data, const cl_bool block, cl_event *wait )
{
    ImageOrigin origin(x, y);
    ImageRegion region(w, h);
    return update(origin, region, data, block, wait);
}

int CLImage::get( const ImageOrigin& origin, const ImageRegion& region, void *data, const cl_bool block, cl_event *wait )
{
    cl_int status= clEnqueueReadImage(ActiveQueue->queue(), m_image, block, origin, region, 0, 0, data, 0, NULL, wait);
    return (status == CL_SUCCESS) ? 0 : -1;
}

int CLImage::get( void *data, const cl_bool block, cl_event *wait )
{
    ImageOrigin origin(0, 0);
    ImageRegion region(m_width, m_height);
    return get(origin, region, data, block, wait);
}

int CLImage::get( const size_t x, const size_t y, const size_t w, const size_t h, void *data, const cl_bool block, cl_event *wait )
{
    ImageOrigin origin(x, y);
    ImageRegion region(w, h);
    return get(origin, region, data, block, wait);
}


CLImage *createImage( const cl_mem_flags flags, const cl_image_format *format, const size_t width, const size_t height, void *data, const size_t pitch )
{
    return CLManager<CLImage>::manager().insert( 
        new CLImage(flags, format, width, height, data, pitch) );
}

CLImage *createReadImage( const cl_image_format *format, size_t width, size_t height, void *data, const size_t pitch )
{
    cl_mem_flags flags= CL_MEM_READ_ONLY;
    if(data != NULL)
        flags= flags | CL_MEM_COPY_HOST_PTR;
    return createImage(flags, format, width, height, data, pitch );
}

CLImage *createWriteImage( const cl_image_format *format, size_t width, size_t height, size_t pitch )
{
    cl_mem_flags flags= CL_MEM_WRITE_ONLY;
    return createImage(flags, format, width, height, NULL, pitch);
}

CLImage *createGLObjectImage( const cl_mem_flags flags, const unsigned int glname, const unsigned int texture_target, const int level )
{
    return CLManager<CLImage>::manager().insert(
        new CLImage(flags, glname, texture_target, level) );
}
    
}       // namespace gk



