
// basic image viewer loading hdr + sdl_image supported formats : png, jpg, bmp, tga images.

#include "nvwidgets/nvSdlWidgets.h"
#include "App.h"

#include "GL/GLPlatform.h"
#include "GLManager.h"
#include "EffectIO.h"
#include "EffectShaderManager.h"
#include "ImageIO.h"
#include "TextureManager.h"
#include "SamplerManager.h"
#include "FramebufferManager.h"
#include "GL/TPSampler.h"
#include "GL/TPTextureUnits.h"
#include "GL/TPShaderProgram.h"



class ImageViewer : public gk::App
{
    std::string m_filename;
    gk::GLTexture2D *m_image;
    gk::GLTexture2D *m_colors;
    gk::GLShaderProgram *m_program;
    gk::GLSampler *m_sampler;
    gk::GLFramebuffer *m_framebuffer;
    
    nv::SdlUIContext m_ui;
    
    float m_compression;
    float m_saturation_max;
    float m_saturation_delta;
    
    float m_saturation;
    float m_saturation_fine;
    int m_mode;
    bool m_show_histogram;
    
    float m_ymin;
    float m_ymax;
    
    float m_bins[256];
    int m_bins_n;
    float m_bins_min;
    float m_bins_max;
    
public:
    ImageViewer( const int w, const int h, const std::string& filename )
        :
        gk::App(w, h),
        m_filename(filename),
        m_image(NULL),
        m_colors(NULL),
        m_program(NULL),
        m_sampler(NULL),
        m_compression(2.f),
        m_saturation_max(1.f),
        m_saturation_delta(.1f),
        m_saturation(1.f),
        m_saturation_fine(0.f),
        m_mode(0),
        m_show_histogram(true),
        m_ymin(0.f),
        m_ymax(1.f),
        m_bins_n(256),
        m_bins_min(0.f),
        m_bins_max(1.f)
    {
        m_ui.init(w, h);
    }

    ~ImageViewer( ) {}

    void processWindowResize( SDL_ResizeEvent& event )
    {
        m_ui.reshape(event.w, event.h);
    }
    
    void processMouseButtonEvent( SDL_MouseButtonEvent& event )
    {
        m_ui.processMouseButtonEvent(event);
    }
    
    void processMouseMotionEvent( SDL_MouseMotionEvent& event )
    {
        m_ui.processMouseMotionEvent(event);
    }
    
    void processKeyboardEvent( SDL_KeyboardEvent& event )
    {
        m_ui.processKeyboardEvent(event);
    }
    
    int init( )
    {
        gk::TextureUnitState::init();
        gk::FramebufferState::init();
        
        // charge la texture de fausses couleurs
        gk::Image *colors= gk::ImageIO::read("false_colors.png");
        if(colors == NULL)
            return -1;
        m_colors= gk::createTexture2D(gk::UNIT0, colors);
        if(m_colors == NULL || m_colors->createGLResource() < 0)
            return -1;
        
        m_sampler= gk::createSampler();
        if(m_sampler == NULL || m_sampler->createGLResource() < 0)
            return -1;
        
        // charge le shader de 'tone' mapping hdr
        gk::Effect *effect= gk::EffectIO::read("hdr_tone.gkfx");
        m_program= gk::EffectShaderManager(effect).createShaderProgram("hdr");
        if(m_program == NULL || m_program->createGLResource() < 0)
            return -1;
        
        // charge l'image
        gk::HDRImage *hdr= gk::HDRImageIO::read(m_filename);
        if(hdr == NULL)
        {
            printf(" -- '%s' failed.\n", m_filename.c_str());
            return -1;
        }
        
        m_image= gk::createTexture2D(gk::UNIT0, hdr);
        if(m_image == NULL || m_image->createGLResource() < 0)
            return -1;
        
        m_framebuffer= gk::createFramebuffer(hdr->width(), hdr->height(), gk::COLOR0_BIT | gk::DEPTH_BIT);
        if(m_framebuffer == NULL || m_framebuffer->createGLResource() < 0)
            return -1;
        
        // modifier le titre de la fenetre
        SDL_WM_SetCaption(m_filename.c_str(), "");
        
        // evaluer les valeurs initiales de compression et saturation
        // . valeurs par defaut
        m_compression= 2.f;
        m_saturation= 100.f;
        m_saturation_max= 100.f;
        m_saturation_delta= 10.f;
        m_saturation_fine= 0.f;
        m_mode= 0.f;
        
        m_bins_n= 256;
        for(int i= 0; i< m_bins_n; i++)
            m_bins[i]= 0.f;
        
        // . trouver les extremess
        float ymin= HUGE_VAL;
        float ymax= -HUGE_VAL;
        float sum= 0.f;
        for(int y= 0; y < hdr->height(); y++)
            for(int x= 0; x < hdr->width(); x++)
            {
                const gk::HDRPixel &pixel= hdr->getPixel(x, y);
                const float l= (pixel.r + pixel.g + pixel.b) / 3.f;
                
                if(l < ymin)
                    ymin= l;
                if(l > ymax)
                    ymax= l;
                
                sum= sum + l;
            }
        
        m_ymin= ymin;
        m_ymax= ymax;
            
        printf("min %f < %f < max %f\n", 
            ymin, sum / (hdr->width() * hdr->height()), ymax);
        
        // . construit l'histogramme
        const float bin_scale= (float) m_bins_n / (ymax - ymin);
        for(int y= 0; y < hdr->height(); y++)
            for(int x= 0; x < hdr->width(); x++)
            {
                const gk::HDRPixel &pixel= hdr->getPixel(x, y);
                const float l= (pixel.r + pixel.g + pixel.b) / 3.f;
                
                int k= (l - ymin) * bin_scale;
                if(k < 0)
                    k= 0;
                if(k >= m_bins_n)
                    k= m_bins_n -1;
                
                m_bins[k]+= 1.f;
            }
        
        m_bins_min= hdr->width() * hdr->height();
        m_bins_max= 0;
        float bins_sum= 0.f;
        const float bin_normalize= 1.f / (hdr->width() * hdr->height());
        for(int i= 0; i < m_bins_n; i++)
        {
            m_bins[i]= m_bins[i] * bin_normalize;
            
            if(m_bins[i] < m_bins_min)
                m_bins_min= m_bins[i];
            if(m_bins[i] > m_bins_max)
                m_bins_max= m_bins[i];

            bins_sum+= m_bins[i];
        }
        
        // . reglage de la dynamique de l'historgramme
        m_bins_max= bins_sum / m_bins_n;
        
        // . reglage des plages des sliders
        m_saturation= ymax;
        m_saturation_max= ymax;
        m_saturation_delta= ymax / 10.f;
        m_saturation_fine= 0.f;
        m_mode= 0.f;
        return 0;
    }
    
    int draw( )
    {
        if(key(SDLK_ESCAPE))
            Close();
        
        if(key('n'))
            resizeWindow(m_image->width(), m_image->height());
        
        if(key(SDLK_SPACE))
        {
            m_show_histogram= !m_show_histogram;
            key(SDLK_SPACE)= 0;
        }
        
        const float x= 0.f;
        const float y= 0.f;
        const float z= -.5f;
        const float w= m_image->width();
        const float h= m_image->height();

        glMatrixMode(GL_PROJECTION);
        glLoadIdentity();
        glOrtho(0., w, 0., h, -1., 1.);
        
        gk::setFramebuffer(m_framebuffer);
        //~ glViewport(0, 0, windowWidth(), windowHeight());
        glViewport(0, 0, m_framebuffer->width(), m_framebuffer->height());
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        
        glMatrixMode(GL_MODELVIEW);
        glLoadIdentity();
        
        gk::setShaderProgram(m_program);
        gk::setUniform(m_program->uniform("compression"), m_compression);
        gk::setUniform(m_program->uniform("saturation"), m_saturation + m_saturation_fine);
        
        if(m_mode == 1)
            gk::setUniform(m_program->uniform("heat"), 1.f);
        else
            gk::setUniform(m_program->uniform("heat"), 0.f);
        
        gk::setTextureUnit(m_program->sampler("image"), m_image, m_sampler);
        gk::setTextureUnit(m_program->sampler("colors"), m_colors, m_sampler);

        glColor3f(1.f, 0.f, 1.f);
        glBegin(GL_QUADS);
            glTexCoord2f(0.f, 1.f);
            glVertex3f(x, y+h, z);
            
            glTexCoord2f(0.f, 0.f);
            glVertex3f(x, y, z);
            
            glTexCoord2f(1.f, 0.f);
            glVertex3f(x+w, y, z);
            
            glTexCoord2f(1.f, 1.f);
            glVertex3f(x+w, y+h, z);
        glEnd();
        
        gk::resetTextureUnit(m_program->sampler("image"));
        gk::resetTextureUnit(m_program->sampler("colors"));
        gk::resetShaderProgram();
        gk::resetFramebuffer();
        
        // copie le framebuffer dans la fenetre
        gk::setReadFramebuffer(m_framebuffer);
        gk::BlitFramebuffer(0, 0, m_framebuffer->width(), m_framebuffer->height(), 0, 0, windowWidth(), windowHeight());
        gk::resetReadFramebuffer();
        glViewport(0, 0, windowWidth(), windowHeight());

        if(m_show_histogram)
        {
            glColor3f(1.f, 0.f, 0.f);
            glLineWidth(2.f);
            glEnable(GL_LINE_SMOOTH);
            glBegin(GL_LINES);
            for(int i= 0; i < m_bins_n; i++)
            {
                glVertex2f(10.f + (float) i / (float) m_bins_n * (float) (w / 2), 10.f);
                glVertex2f(10.f + (float) i / (float) m_bins_n * (float) (w / 2), 10.f + m_bins[i] / m_bins_max * (float) h / 2.f);
            }
            glEnd();
            
            // visualise compression dans l'histogramme
            float bin_scale= (float) m_bins_n / (m_ymax - m_ymin);
            float saturation_height= (float) h;
            
            int saturation_bin= (m_saturation + m_saturation_fine - m_ymin) * bin_scale;
            glColor3f(1.f, 1.f, 0.f);
            glBegin(GL_LINES);
                glVertex2f(10.f + (float) saturation_bin / (float) m_bins_n * (float) (w / 2), 10.f);
                glVertex2f(10.f + (float) saturation_bin / (float) m_bins_n * (float) (w / 2), 10.f + saturation_height);
            glEnd();
            
            // trace la courbe de compression
            glColor3f(0.f, 0.f, 1.f);
            glBegin(GL_LINES);
            for(int i= 0; i < m_bins_n; i++)
            {
                float k1= 1.0 / powf(m_saturation + m_saturation_fine, 1.f / m_compression); // normalisation : saturation == blanc
                float y= (float) (i+1) / (float) m_bins_n * (m_ymax - m_ymin) + m_ymin;
                float color= y * k1 * powf(y, 1.f / m_compression);
                
                glVertex2f(10.f + (float) i / (float) m_bins_n * (float) (w / 2), 10.f + color * h/2 );
            }
            glEnd();
        }
        
        bool do_screenshot= false;
        m_ui.begin();
            m_ui.beginGroup(nv::GroupFlags_GrowDownFromLeft);
                m_ui.beginGroup(nv::GroupFlags_GrowRightFromTop);
                    m_ui.doLabel(nv::Rect(), "saturation");
                    {
                        char tmp[1024];
                        sprintf(tmp, "%.6f", m_saturation + m_saturation_fine);
                        
                        int edit= 0;
                        if(m_ui.doLineEdit(nv::Rect(), tmp, sizeof(tmp), &edit))
                        {
                            float value= m_saturation + m_saturation_fine;
                            if(sscanf(tmp, "%f", &value) == 1)
                            {
                                m_saturation= value;
                                m_saturation_fine= 0.f;
                            }
                        }
                    }
                    
                    if(m_ui.doHorizontalSlider(nv::Rect(0,0, 200, 0), 0.f, m_saturation_max * 4.f, &m_saturation))
                        m_saturation_fine= 0.f;
                    
                    m_ui.doHorizontalSlider(nv::Rect(), 0.f, m_saturation_delta, &m_saturation_fine);
                    m_ui.doCheckButton(nv::Rect(), "histogram", &m_show_histogram);
                    
                    if(m_ui.doButton(nv::Rect(), "save"))
                        do_screenshot= true;
                m_ui.endGroup();
                    
                m_ui.beginGroup(nv::GroupFlags_GrowRightFromTop);
                    m_ui.doRadioButton(1, nv::Rect(), "heat", &m_mode);
                    m_ui.doRadioButton(0, nv::Rect(), "compression", &m_mode);
                    
                    if(m_mode == 0)
                    {
                        char tmp[1024];
                        sprintf(tmp, "%.2f", m_compression);
                        
                        int edit= 0;
                        if(m_ui.doLineEdit(nv::Rect(), tmp, sizeof(tmp), &edit))
                        {
                            float value= m_compression;
                            if(sscanf(tmp, "%f", &value) == 1)
                                m_compression= value;
                        }
                        
                        m_ui.doHorizontalSlider(nv::Rect(0,0, 200, 0), 0.f, 10.f, &m_compression);
                    }
                m_ui.endGroup();
            m_ui.endGroup();
        m_ui.end();

        if(do_screenshot)
        {
            const int width= windowWidth();
            const int height= windowHeight();
            Uint8 *data= new Uint8[width * height * 3];
            assert(data != NULL);
            
            // flip de l'image : Y inverse entre GL et BMP
            Uint8 *flip= data;
            for(int y= height -1; y >= 0; y--)
            {
                glReadPixels(0, y, width, 1, 
                    GL_RGB, GL_UNSIGNED_BYTE, flip);
                flip+= width * 3;
            }
            
            SDL_Surface *bmp= SDL_CreateRGBSurfaceFrom(data, 
                width, height, 24, width * 3, 
        #if 0
                0xFF000000,
                0x00FF0000, 
                0x0000FF00,
                0x00
        #else
                0x000000FF,
                0x0000FF00,
                0x00FF0000, 
                0x00
        #endif
                );
            
            const char *tmp= "output_tone.bmp";
            printf("writing '%s'...\n", tmp);
            SDL_SaveBMP(bmp, tmp);
            SDL_FreeSurface(bmp);
            delete [] data;
        }
        
        SDL_GL_SwapBuffers();
        return 1;
    }
};

int main( int argc, char **argv )
{
    if(argc != 2)
    {
        printf("%s image.hdr\n\n", argv[0]);
        return 0;
    }
    
    ImageViewer viewer(1024, 768, argv[1]);
    viewer.run();
    
    return 0;
}


