#include <cstdlib>
#include <cassert>

#include <vector>

#include "Geometry.h"
#include "Transform.h"
#include "PNTriangle.h"
#include "Mesh.h"
#include "MeshIO.h"
#include "IOFileSystem.h"

//! representation des points de contole d'un triangle de bezier, un pn triangle, cf http://alex.vlachos.com/graphics/CurvedPNTriangles.pdf,
//! a comparer avec phong tesselation cf http://perso.telecom-paristech.fr/~boubek/papers/PhongTessellation/
class PNControlPoints
{
    gk::Point b[4][4][4];
    gk::Normal bn[3][3][3];
    
    float nvij( const gk::Point& pi, const gk::Normal& ni, 
        const gk::Point& pj, const gk::Normal& nj )
    {
        gk::Vector v(pi, pj);
        
        return 2.f * gk::Dot(v, ni + nj) / gk::Dot(v, v);
    }
    
    // projection de la normale sur le plan (pi, ni)
    gk::Normal nhij( const gk::Point& pi, const gk::Normal& ni, 
        const gk::Point& pj, const gk::Normal& nj )
    {
        const float v= nvij(pi, ni, pj, nj);
        return gk::Normalize(ni + nj - gk::Normal(v * (pj - pi)));
    }
    
    // construction des points de controle du triangle de bezier
    void bezier( const gk::PNTriangle& triangle )
    {
        // base d'interpolation pour les sommets.
        b[3][0][0]= triangle.a; // w
        b[0][3][0]= triangle.b; // u
        b[0][0][3]= triangle.c; // v
        
        const float w12= Dot( gk::Vector(triangle.a, triangle.b), triangle.na );
        b[2][1][0]= (2.f * triangle.a + triangle.b - w12 * gk::Vector(triangle.na)) / 3.f;
        
        const float w21= Dot( gk::Vector(triangle.b, triangle.a), triangle.nb );
        b[1][2][0]= (2.f * triangle.b + triangle.a - w21 * gk::Vector(triangle.nb)) / 3.f;
        
        const float w23= Dot( gk::Vector(triangle.b, triangle.c), triangle.nb );
        b[0][2][1]= (2.f * triangle.b + triangle.c - w23 * gk::Vector(triangle.nb)) / 3.f;
        
        const float w32= Dot( gk::Vector(triangle.c, triangle.b), triangle.nc );
        b[0][1][2]= (2.f * triangle.c + triangle.b - w32 * gk::Vector(triangle.nc)) / 3.f;
        
        const float w31= Dot( gk::Vector(triangle.c, triangle.a), triangle.nc );
        b[1][0][2]= (2.f * triangle.c + triangle.a - w31 * gk::Vector(triangle.nc)) / 3.f;
        
        const float w13= Dot( gk::Vector(triangle.a, triangle.c), triangle.na );
        b[2][0][1]= (2.f * triangle.a + triangle.c - w13 * gk::Vector(triangle.na)) / 3.f;
        
        const gk::Point e= (b[2][1][0] + b[1][2][0] + b[0][2][1] + b[0][1][2] + b[1][0][2] + b[2][0][1]) / 6.f;
        const gk::Point v= (triangle.a + triangle.b + triangle.c) / 3.f;
        b[1][1][1]= e + (e - v) / 2.f;
        
        // base d'interpolation pour les normales
        bn[2][0][0]= triangle.na;
        bn[0][2][0]= triangle.nb;
        bn[0][0][2]= triangle.nc;
        
        bn[1][1][0]= nhij(triangle.a, triangle.na, triangle.b, triangle.nb);
        bn[0][1][1]= nhij(triangle.b, triangle.nb, triangle.c, triangle.nc);
        bn[1][0][1]= nhij(triangle.c, triangle.nc, triangle.a, triangle.nc);
    }
    
public:
    //! evalue un point (u, v) sur la surface : p(u, v) = w*a + u*b + v*c, avec w= 1-u-v
    gk::Point point( const float u, const float v ) const
    {
        const float u2= u * u;
        const float u3= u2 * u;
        const float v2= v * v;
        const float v3= v2 * v;
        const float w= 1.f - u - v;
        const float w2= w * w;
        const float w3= w2 * w;
        
        gk::Point p= b[3][0][0] * w3 + b[0][3][0] * u3 + b[0][0][3] * v3;
        p+= b[2][1][0] * 3.f * w2 * u + b[1][2][0] * 3.f * w * u2 + b[2][0][1] * 3.f * w2 * v;
        p+= b[0][2][1] * 3.f * u2 * v + b[1][0][2] * 3.f * w * v2 + b[0][1][2] * 3.f * u * v2;
        p+= b[1][1][1] * 6.f * w * u * v;
        
        return p;
    }
    
    //! evalue une normale sur la surface : n(u, v)= w*na + u*nb + v*nc, avec w= 1-u-v
    gk::Normal normal( const float u, const float v ) const
    {
        const float u2= u * u;
        const float v2= v * v;
        const float w= 1.f - u - v;
        const float w2= w * w;
        
        gk::Normal n= bn[2][0][0] * w2 + bn[0][2][0] * u2 + bn[0][0][2] * v2;
        n+= bn[1][1][0] * w * u + bn[0][1][1] * u * v + bn[1][0][1] * w * v;
        
        return gk::Normalize(n);
    }
    
    //! construction des points de controle du pn triangle.
    PNControlPoints( const gk::PNTriangle& triangle )
    {
        bezier(triangle);
    }
};

//! representation d'un triangle sur le domaine parametrique (u, v).
struct UVTriangle
{
    gk::Point2 a, b, c;
    
    //! constructeur par defaut : construit un triangle unitaire.
    UVTriangle( )
        :
        a( gk::Point2(0.f, 0.f) ),
        b( gk::Point2(1.f, 0.f) ),
        c( gk::Point2(0.f, 1.) )
    {}
    
    //! construit le triangle 'geometrique' associe.
    gk::PNTriangle eval( const PNControlPoints& cage ) const
    {
        gk::PNTriangle triangle;
        triangle.a= cage.point(a.x, a.y);
        triangle.na= cage.normal(a.x, a.y);
        triangle.b= cage.point(b.x, b.y);
        triangle.nb= cage.normal(b.x, b.y);
        triangle.c= cage.point(c.x, c.y);
        triangle.nc= cage.normal(c.x, c.y);
        
        return triangle;
    }
    
    //! decoupage regulier du triangle en 4 sous triangles.
    void subdivide( UVTriangle& sub1, UVTriangle& sub2, UVTriangle& sub3, UVTriangle& sub4 ) const
    {
        sub1.a= a;
        sub1.b= (a + b) / 2.f;
        sub1.c= (c + a) / 2.f;
        
        sub2.a= (c + a) / 2.f;
        sub2.b= (a + b) / 2.f;
        sub2.c= (b + c) / 2.f;
        
        sub3.a= (b + c) / 2.f;
        sub3.b= (a + b) / 2.f;
        sub3.c= b;
        
        sub4.a= (c + a) / 2.f;
        sub4.b= (b + c) / 2.f;
        sub4.c= c;
    }
    
    //! construit le triangle 'geometrique' pour les sommets a, b, c.
    static
    gk::PNTriangle eval( const PNControlPoints& cage, const gk::Point2& a, const gk::Point2& b, const gk::Point2& c )
    {
        gk::PNTriangle triangle;
        triangle.a= cage.point(a.x, a.y);
        triangle.na= cage.normal(a.x, a.y);
        triangle.b= cage.point(b.x, b.y);
        triangle.nb= cage.normal(b.x, b.y);
        triangle.c= cage.point(c.x, c.y);
        triangle.nc= cage.normal(c.x, c.y);
        
        return triangle;
    }
};


void subdivide( const PNControlPoints& cage, const UVTriangle& sub, std::vector<gk::PNTriangle>& out, const int level = 1 )
{
    bool valid= false;
    
    // estimer la qualite de la subdivision == distance max entre le milieu du triangle et le point associe sur la surface
#if 0
    gk::Point2 subm= (sub.a + sub.b + sub.c) / 3.f;
    gk::Point sm= cage.point(subm.x, subm.y);
    gk::Point m= (cage.point(sub.a.x, sub.a.y) + cage.point(sub.b.x, sub.b.y) + cage.point(sub.c.x, sub.c.y)) / 3.f;
    if(gk::Distance(sm, m) < 0.03f)
        valid= true;
#endif
    
    if(level <= 0 || valid)
    {
        out.push_back( sub.eval(cage) );
        return;
    }
    
    UVTriangle sub1, sub2, sub3, sub4;
    sub.subdivide(sub1, sub2, sub3, sub4);
    
    subdivide(cage, sub1, out, level -1);
    subdivide(cage, sub2, out, level -1);
    subdivide(cage, sub3, out, level -1);
    subdivide(cage, sub4, out, level -1);
}

void subdivide( const gk::PNTriangle& triangle, std::vector<gk::PNTriangle>& out, const int level = 1 )
{
    // subdivision parametrique
    PNControlPoints cage(triangle);
    
    UVTriangle sub;
    subdivide(cage, sub, out, level);
}

int pntriangles_subdivide( const std::string& filename, gk::Mesh *model, const int level= 1 )
{
    if(model == NULL)
        return -1;
    
    // 1. recupere les triangles et leurs normales
    std::vector<gk::PNTriangle> out;
    const int triangles_n= model->triangleCount();
    for(int i= 0; i < triangles_n; i++)
        // 2. subdivise chaque triangle
        subdivide(model->getPNTriangle(i), out, level);
    
    // 2. ecrire le fichier .obj
    char tmp[1024];
    sprintf(tmp, "%s.%d.obj", gk::IOFileSystem::basename(filename).c_str(), level);
    printf("writing mesh '%s'...\n", tmp);
    FILE *obj= fopen(tmp, "wt");
    if(obj == NULL)
    {
        printf("error creating file.\nfailed.\n");
        return -1;
    }

    // ecrire les positions
    const int count= (int) out.size();
    for(int i= 0; i < count; i++)
    {
        gk::PNTriangle& triangle= out[i];
        
        // positions
        fprintf(obj, "v %f %f %f\n", triangle.a.x, triangle.a.y, triangle.a.z);
        fprintf(obj, "v %f %f %f\n", triangle.b.x, triangle.b.y, triangle.b.z);
        fprintf(obj, "v %f %f %f\n", triangle.c.x, triangle.c.y, triangle.c.z);
        
        // normales
        fprintf(obj, "vn %f %f %f\n", triangle.na.x, triangle.na.y, triangle.na.z);
        fprintf(obj, "vn %f %f %f\n", triangle.nb.x, triangle.nb.y, triangle.nb.z);
        fprintf(obj, "vn %f %f %f\n", triangle.nc.x, triangle.nc.y, triangle.nc.z);
        
        // triangle
        fprintf(obj, "f -3//-3 -2//-2 -1//-1\n");
    }
    
    fclose(obj);
    printf("done.\n");

    return 0;
}

int main( int argc, char **argv )
{
    if(argc != 4)
    {
        printf("usage: %s model.obj level output.obj\n", argv[0]);
        return 1;
    }
    
    // charge un objet
    gk::Mesh *model= gk::MeshIO::read(argv[1]);
    
    // subdivision pn triangles
    int level= 1;
    if(sscanf(argv[2], "%d", &level) != 1)
    {
        printf("usage: %s model.obj level output.obj\n", argv[0]);
        return 1;
    }
    
    if(pntriangles_subdivide(argv[3], model, level) < 0)
        return 1;
    
    return 0;
}
    
