
#include "GL/GLPlatform.h"
#include "IOFileSystem.h"

#include "EffectIO.h"
#include "EffectShaderManager.h"

#include "Mesh.h"
#include "MeshIO.h"
#include "BufferManager.h"

#include "App.h"

#include "Camera.h"
#include "Orbiter.h"


class TP : public gk::App
{
    gk::Effect *m_effect;
    gk::GLShaderProgram *m_program;
    gk::GLAttributeBuffer *m_positions;
    gk::GLAttributeBuffer *m_normals;
    gk::GLIndexBuffer *m_index;
    
    std::string m_filename;
    gk::FirstPersonCamera m_camera;
    gk::Orbiter m_orbiter;
    
public:
    TP( const std::string& filename, const int w= 1024, const int h= 768 )
        :
        gk::App(w, h),
        m_effect(NULL),
        m_program(NULL),
        m_positions(NULL),
        m_normals(NULL),
        m_index(NULL),
        m_filename(filename),
        m_camera(),
        m_orbiter()
    {}
    
    ~TP( ) {}
    
    int init( )         // charger les objets, les vertex buffers, etc.
    {
        gk::BufferState::init();
        
        // init camera
        m_camera= gk::FirstPersonCamera(50.f, 1.f, 1.f, 1000.f, windowWidth(), windowHeight());
        m_camera.move( gk::Point(0.f, 0.f, 50.f) );
        
        // init object manipulator
        m_orbiter= gk::Orbiter();
        
        m_effect= gk::EffectIO::read("basic.gkfx");
        if(m_effect == NULL)
            return -1;
        
        m_program= gk::EffectShaderManager(m_effect).createShaderProgram("program");
        if(m_program == NULL || m_program->createGLResource() < 0)
            return -1;
        
        gk::Mesh *mesh= gk::MeshIO::read(m_filename);
        if(mesh == NULL)
            return -1;
        
        if(mesh->normalCount() != mesh->positionCount())
            mesh->buildNormals();
        
        m_positions= gk::createAttributeBuffer(
            mesh->positionCount(), 
            mesh->positionCount() * sizeof(float [3]), &mesh->positions().front());
        if(m_positions == NULL || m_positions->createGLResource() < 0)
            return -1;
        
        m_normals= gk::createAttributeBuffer(
            mesh->normalCount(),
            mesh->normalCount() * sizeof(float [3]), &mesh->normals().front());
        if(m_normals == NULL || m_normals->createGLResource() < 0)
            return -1;
        
        if(mesh->indiceCount() > 0)
        {
            m_index= gk::createIndexBuffer(
                mesh->indiceCount(), 
                mesh->indiceCount() * sizeof(int), &mesh->indices().front());
            if(m_index == NULL || m_index->createGLResource() < 0)
                return -1;
        }
    
        return 0;       // tout c'est bien passe
    }
    
    int quit( )         
    {
        // toutes les ressources seront liberees par shaderManager et bufferManager, etc.
        return 0;
    }
    
    int draw( )
    {
        if(key(SDLK_ESCAPE))
            // sortir si l'utilisateur appuye sur ESC
            Close();

        // camera
        m_camera.setViewport(windowWidth(), windowHeight());
        
        // controle de la camera a la souris
        int x, y;
        int button= SDL_GetRelativeMouseState(&x, &y);
        if(button & SDL_BUTTON(1))
        {
            // move camera
            m_camera.rotateUp(-x);
            m_camera.rotateRight(-y);
        }
        if(button & SDL_BUTTON(3))
        {
            // re-orient object
            m_orbiter.rotateUp(x);
            m_orbiter.rotateRight(y);
        }
        
        // controle au clavier de la camera
        if(SDL_GetModState() & KMOD_CTRL)
        {
            if(key(SDLK_UP))
                m_camera.rotateRight(1.f);
            if(key(SDLK_DOWN))
                m_camera.rotateRight(-1.f);
            if(key(SDLK_LEFT))
                m_camera.rotateUp(1.f);
            if(key(SDLK_RIGHT))
                m_camera.rotateUp(-1.f);
        }
        else
        {
            if(key('z') || key(SDLK_UP))
                m_camera.moveForward(-1.f);
            if(key('s') || key(SDLK_DOWN))
                m_camera.moveForward(1.f);
            if(key('q') || key(SDLK_LEFT))
                m_camera.moveRight(-1.f);
            if(key('d') || key(SDLK_RIGHT))
                m_camera.moveRight(1.f);
            if(key(SDLK_PAGEUP))
                m_camera.moveUp(1.f);
            if(key(SDLK_PAGEDOWN))
                m_camera.moveUp(-1.f);
            
            if(key('c'))        // copier / coller la position de la camera
            {
                key('c')= 0;
                m_camera.write(m_filename);
            }
            if(key('v'))
            {
                key('v')= 0;
                m_camera.read(m_filename);
            }
            
            // modifier la vitesse de deplacement
            if(key(SDLK_PLUS))
                m_camera.setSpeed(m_camera.speed() * 1.1f);
            if(key(SDLK_MINUS))
                m_camera.setSpeed(m_camera.speed() * .9f);
        }
        
        // redimensionner la fenetre, si necessaire.
        glViewport(0, 0, windowWidth(), windowHeight());
        // effacer le buffer de dessin.
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        
        // dessiner quelque chose.
        gk::setShaderProgram(m_program);

        gk::Transform model= m_orbiter.transform();
        gk::Transform view= m_camera.view();
        gk::Transform projection= m_camera.projection();
        gk::Transform mvp= projection * view * model;
        
        // passer la matrice model view projection au shader
        gk::setUniform(m_program->uniform("mvpMatrix"), mvp.matrix());
        
        // parametrer les autres uniforms du shader
        // recuperer la valeur dans le script
        gk::Vec4Value color= m_effect->find("uniform.color")->asVec4();
        gk::setUniform(m_program->uniform("color"), color.x, color.y, color.z, color.w);

        // activer les buffers d'attributs de sommets
        gk::setVertexBuffer(m_program->attribute("position"), m_positions, 3, GL_FLOAT);
        gk::setVertexBuffer(m_program->attribute("normal"), m_normals, 3, GL_FLOAT);
        if(m_index != NULL)
        {
            gk::setIndexBuffer(m_index, GL_UNSIGNED_INT);
            // draw indexe, utilser le buffer d'indexation
            gk::DrawElements(GL_TRIANGLES);
        }
        else
        {
            // pas d'index, dessiner tout les points
            gk::DrawArrays(GL_TRIANGLES, 0, m_positions->count());
        }
        
        // afficher le dessin.
        SDL_GL_SwapBuffers();
        // continuer.
        return 1;
    }
};


int main( int argc, char *argv[] )
{
    if(argc < 2)
    {
        printf("usage: %s object.obj\n\n", argv[0]);
        return 1;
    }
    
    TP app(argv[1]);
    app.run();
    
    return 0;
}
