
#ifndef _GK_ORIENTED_FRAME_H
#define _GK_ORIENTED_FRAME_H

#include "Transform.h"

namespace gk {

//! manipulation d'une Transform : passage d'un repere '2' vers '1'.
class OrientedFrame : public Transform
{
public:
    //! utilisation interne. constructeur.
    OrientedFrame( const Transform& transform )
    {
        m= transform.matrix();
        mInv= transform.inverseMatrix();
    }
    
    //! constructeur.
    OrientedFrame( ) 
        :
        Transform()
    {}
    
    //! desctructeur.
    ~OrientedFrame( ) {}
    
    //! \name origine et axes du repere '2' exprimes dans le repere '1'.
    // @{
    //! renvoie la position de l'origine, dans le repere '1'.
    const Point origin( ) const
    {
        return (*this)( Point(0.f, 0.f, 0.f) );
    }
    
    //! renvoie le vecteur 'vers le haut', dans le repere '1'
    const Vector up( ) const // Y
    {
        return (*this)( Vector(0.f, 1.f, 0.f) );
    }
    
    //! renvoie le vecteur 'vers la droite', dans le repere '1'.
    const Vector right( ) const      // X
    {
        return (*this)( Vector(1.f, 0.f, 0.f) );
    }
    
    //! renvoie le vecteur 'devant', dans le repere '1'.
    const Vector forward( ) const    // Z
    {
        return (*this)( Vector(0.f, 0.f, 1.f) );
    }
    // @}
    
    //! remplace la transformation.
    void setTransform( const Transform& transform )
    {
        *this= transform;
    }
    
    //! \name manipulation de la transformation repere '2' vers '1', les transformations elementaires se composent avec la transformation courante.
    //! utiliser identity( ) pour la reinitialiser.
    
    // @{
    //! deplacement en avant / arriere, compose la translation avec la transformation courante.
    void moveForward( const float v )
    {
        *this= (*this) * Translate( Vector(0.f, 0.f, v) );
    }
    
    //! deplacement a gauche / droite, compose la translation avec la transformation courante.
    void moveRight( const float v )
    {
        *this= (*this) * Translate( Vector(v, 0.f, 0.f) );
    }
    
    //! deplacement haut / bas, compose la translation avec la transformation courante.
    void moveUp( const float v )
    {
        *this= (*this) * Translate( Vector(0.f, v, 0.f) );
    }
    
    //! deplacement, compose la translation avec la transformation courante.
    void move( const Vector& v )
    {
        *this= (*this) * Translate(v);
    }
    
    //! rotation autour de la verticale, compose la rotation avec la transformation courante.
    void rotateUp( const float angle )
    {
        *this= (*this) * RotateY(angle);
    }
    
    //! rotation autour de l'horizontale, compose la rotation avec la transformation courante.
    void rotateRight( const float angle )
    {
        *this= (*this) * RotateX(angle);
    }
    
    //! rotation autour de l'axe optique, compose la rotation avec la transformation courante.
    void rotateForward( const float angle )
    {
        *this= (*this) * RotateZ(angle);
    }

    //! vise un point dans le repere local, compose la transformation avec la transformation courante.
    void lookAt( const Point& p )
    {
        // determine un vecteur up 'automatiquement' == non colineaire avec z
        Vector up;
        
        // cf. CoordinateSystem( )
        if ( fabsf( p.x ) > fabsf( p.y ) )
        {
            float invLen = 1.f / sqrtf( p.x * p.x + p.z * p.z );
            up= Vector( -p.z * invLen, 0.f, p.x * invLen );
        }
        else
        {
            float invLen = 1.f / sqrtf( p.y * p.y + p.z * p.z );
            up= Vector( 0.f, p.z * invLen, -p.y * invLen );
        }
        
        *this= (*this) * LookAt( Point(0.f, 0.f, 0.f), p, up);
    }
    
    //! vise un point dans le repere local, compose la transformation avec la transformation courante.
    void lookAt( const Point& p, const Vector& up )
    {
        *this= (*this) * LookAt( Point(0.f, 0.f, 0.f), p, up);
    }
    
    //! vise un point dans le repere global, compose la transformation avec la transformation courante.
    void lookAtWorld( const Point& p )
    {
        // determine un vecteur up 'automatiquement' == non colineaire avec z
        Point origin= inverse( Point(0.f, 0.f, 0.f) );
        Vector up(origin, p);
        
        // cf. CoordinateSystem( )
        if ( fabsf( up.x ) > fabsf( up.y ) )
        {
            float invLen = 1.f / sqrtf( up.x * up.x + up.z * up.z );
            up= Vector( -up.z * invLen, 0.f, up.x * invLen );
        }
        else
        {
            float invLen = 1.f / sqrtf( up.y * up.y + up.z * up.z );
            up= Vector( 0.f, up.z * invLen, -up.y * invLen );
        }
        
        // retrouve l'origine du repere dans le repere 'parent' / world
        *this= (*this) * LookAt(origin, p, up);
    }
    
    //! vise un point dans le repere global, compose la transformation avec la transformation courante.
    void lookAtWorld( const Point& p, const Vector& up )
    {
        // retrouve l'origine du repere dans le repere 'parent' / world
        *this= (*this) * LookAt(inverse( Point(0.f, 0.f, 0.f) ), p, up);
    }
    
    //! repositionne completement la camera, compose la transformation avec la transformation courante.
    void lookAt( const Point& eye, const Point& p )
    {
        // determine un vecteur up 'automatiquement' == non colineaire avec z
        Point origin= inverse( Point(0.f, 0.f, 0.f) );
        Vector up(origin, p);
        
        // cf. CoordinateSystem( )
        if ( fabsf( up.x ) > fabsf( up.y ) )
        {
            float invLen = 1.f / sqrtf( up.x * up.x + up.z * up.z );
            up= Vector( -up.z * invLen, 0.f, up.x * invLen );
        }
        else
        {
            float invLen = 1.f / sqrtf( up.y * up.y + up.z * up.z );
            up= Vector( 0.f, up.z * invLen, -up.y * invLen );
        }
        
        *this= (*this) * LookAt(eye, p, up);
    }
    
    //! repositionne completement la camera, compose la transformation avec la transformation courante.
    void lookAt( const Point& eye, const Point& p, const Vector& up )
    {
        *this= (*this) * LookAt(eye, p, up);
    }
    
    //! decompose la transformation en rotations sur z/x/y.
    //! cf. real time rendering 3, p 68
    void decompose( float& forward, float& right, float& up ) const
    {
        up= Degrees( atan2f( -m.m[2][0], m.m[2][2] ) );
        right= Degrees( asinf( m.m[2][1] ) );
        if(m.m[0][1] == 0.f && m.m[1][1] == 0.f)
            forward= Degrees( atan2f( m.m[1][0], m.m[0][0] ) );
        else
            forward= Degrees( atan2f( -m.m[0][1], m.m[1][1] ) );
        
    #if 0
        printf("OrientedFrame( )::decompose( )\nT=\n");
        m.print();
        printf("Rz= %f, Rx= %f, Ry= %f\n", forward, right, up);
        
        OrientedFrame tmp;
        tmp.rotateUp(up);
        tmp.rotateRight(right);
        tmp.rotateForward(forward);
        tmp.print();
        printf("\n");
    #endif
    }
    // @}
    
    //! renvoie la transformation.
    const Transform& transform( ) const
    {
        return *this;
    }
    
    //! renvoie la matrice representant la transformation directe, passage du repere '2' vers '1'.
    const Matrix4x4& matrix( ) const
    {
        return m;
    }
    
    //! renvoie la matrice representation la transformation inverse, passage du repere '1' vers '2'.
    const Matrix4x4& inverseMatrix( ) const
    {
        return mInv;
    }
};

}

#endif
