
#ifndef _GK_TRIANGLE_H
#define _GK_TRIANGLE_H

#include "Geometry.h"
#include "Transform.h"

namespace gk {
    
//! representation d'un triangle 'geometrique'.
struct Triangle
{
    Point a, b, c;
    
    //! constructeur par defaut.
    Triangle( ) {}
    
    //! construit un triangle connaissant ses 3 sommets.
    Triangle( const Point& a, const Point& b, const Point& c )
        :
        a(a),  b(b), c(c)
    {}
    
    //! destructeur.
    ~Triangle( ) {}
    
    float getArea( ) const
    {
        Vector ab(a, b);
        Vector ac(a, c);
        
        return .5f * Cross(ab, ac).Length();
    }
    
    Normal getNormal( ) const
    {
        Vector ab(a, b);
        Vector ac(a, c);
        
        return Normal( Normalize(Cross(ab, ac)) );
    }
    
    BBox getBBox( ) const
    {
        BBox bbox;
        bbox.Union(a);
        bbox.Union(b);
        bbox.Union(c);
        
        return bbox;
    }
    
    //! renvoie un triangle transforme par 't'.
    Triangle transform( const Transform& t )
    {
        return Triangle( t(a), t(b), t(c) );
    }
    
    //! intersection avec un rayon.
    //! renvoie faux s'il n'y a pas d'intersection, une intersection peut exister mais peut ne pas se trouver dans [tmin tmax] du rayon.
    //! renvoie vrai + les coordonnees barycentriques du point d'intersection + sa position le long du rayon.
    //! utiliser Mesh::getUVNormal() et Mesh::getUVTexCoord() pour interpoler les attributs du point d'intersection.
    bool Intersect( const Ray &ray, const float htmin, const float htmax, 
        float &rt, float &ru, float&rv ) const
    {
        /* begin calculating determinant - also used to calculate U parameter */
        Vector ac(a, c);
        const Vector pvec= Cross(ray.d, ac);
        
        /* if determinant is near zero, ray lies in plane of triangle */
        Vector ab(a, b);
        const float det= Dot(ab, pvec);
        if (det > -EPSILON && det < EPSILON)
            return false;
        
        const float inv_det= 1.0f / det;
        
        /* calculate distance from vert0 to ray origin */
        const Vector tvec(a, ray.o);
        
        /* calculate U parameter and test bounds */
        const float u= Dot(tvec, pvec) * inv_det;
        if(u < 0.0f || u > 1.0f)
            return false;
        
        /* prepare to test V parameter */
        const Vector qvec= Cross(tvec, ab);
        
        /* calculate V parameter and test bounds */
        const float v= Dot(ray.d, qvec) * inv_det;
        if(v < 0.0f || u + v > 1.0f)
            return false;
        
        /* calculate t, ray intersects triangle */
        rt= Dot(ac, qvec) * inv_det;
        ru= u;
        rv= v;
        
        // ne renvoie vrai que si l'intersection est valide (comprise entre tmin et tmax du rayon)
        return (rt < htmax && rt > htmin);
    }
    
    //! intersection avec un rayon. 
    //! version C, utilisable avec vec.h et mat44.h.
    //! cf. Intersect( const Ray &ray, ... ) pour les explications detaillees.
    bool Intersect( const float * origin, const float *direction,  
        const float tmin, const float tmax, 
        float *rt, float *ru, float *rv ) const
    {
        assert(rt != NULL);
        assert(ru != NULL);
        assert(rv != NULL);
        
        Ray ray(
            Point(origin[0], origin[1], origin[2]), 
            Vector(direction[0], direction[1], direction[2]), 
            tmin, tmax);
        
        return Intersect(ray, tmin, tmax, *rt, *ru, *rv);
    }
};

}
#endif
