
#ifndef _GK_NAME_H
#define _GK_NAME_H

#include <string>
#include <cstring>
#include <cassert>


namespace gk {

#define GK_NAME_MAX 44
#define GK_NAME_HASH_SEED 1

//~ #ifdef GK_NAME_NOSTRING
    //~ #warning Name == hash code
//~ #endif    
    
//! representation d'un identifiant par une chaine de caracteres et sa valeur de hachage.
//! definir GK_NAME_NOSTRING pour la version optimisee qui ne stocke que la valeur de hachage : attention c_str() n'est plus accessible !!
class Name
{
#ifndef GK_NAME_NOSTRING
    char m_name[GK_NAME_MAX];
#endif
    unsigned int m_hash;
    
    //! murmurhash2 cf http://sites.google.com/site/murmurhash/
    static
    unsigned int murmur2( const char *string, const int length, const unsigned int seed )
    {
        // 'm' and 'r' are mixing constants generated offline.
        // They're not really 'magic', they just happen to work well.
        const unsigned int m = 0x5bd1e995;
        const int r = 24;

        // Initialize the hash to a 'random' value
        unsigned int h = seed ^ length;
        int len= length;

        // Mix 4 bytes at a time into the hash
        const unsigned char *data = (const unsigned char *) string;
        while(len >= 4)
        {
            unsigned int k = * (const unsigned int *) data;

            k *= m; 
            k ^= k >> r; 
            k *= m; 

            h *= m; 
            h ^= k;

            data += 4;
            len -= 4;
        }

        // Handle the last few bytes of the input array
        switch(len)
        {
            case 3: h ^= data[2] << 16;
            case 2: h ^= data[1] << 8;
            case 1: h ^= data[0];
            h *= m;
        };

        // Do a few final mixes of the hash to ensure the last few
        // bytes are well-incorporated.
        h ^= h >> 13;
        h *= m;
        h ^= h >> 15;

        return h;
    }

    void hash( const char *string )
    {
        m_hash= murmur2(string, GK_NAME_MAX, GK_NAME_HASH_SEED);
    }
    
public:
    //! constructeur par defaut.
    Name( ) 
        :
        m_hash(0)
    {
    #ifndef GK_NAME_NOSTRING
        m_name[0]= 0;
    #endif
    }
    
    //! constructeur.
    Name( const char *string )
    {
    #ifndef GK_NAME_NOSTRING
        strncpy(m_name, string, GK_NAME_MAX);
        m_name[GK_NAME_MAX -1]= 0;
        hash(m_name);

    #else
        char name[GK_NAME_MAX];
        strncpy(name, string, GK_NAME_MAX);
        name[GK_NAME_MAX -1]= 0;
        hash(name);
    #endif
    }
    
    //! constructeur.
    Name( const std::string& string )
    {
    #ifndef GK_NAME_NOSTRING
        strncpy(m_name, string.c_str(), GK_NAME_MAX);
        m_name[GK_NAME_MAX -1]= 0;
        hash(m_name);

    #else
        char name[GK_NAME_MAX];
        strncpy(name, string.c_str(), GK_NAME_MAX);
        name[GK_NAME_MAX -1]= 0;
        hash(name);
    #endif
    }

    //! constructeur.
    Name( const Name& b )
    {
    #ifndef GK_NAME_NOSTRING
        strcpy(m_name, b.m_name);
    #endif
        m_hash= b.m_hash;
    }
    
    //! affectation.
    Name& operator=( const Name&b )
    {
    #ifndef GK_NAME_NOSTRING
        strcpy(m_name, b.m_name);
    #endif
        m_hash= b.m_hash;
        return *this;
    }
    
    unsigned int hash( ) const
    {
        return m_hash;
    }
    
#if 1
    //! renvoie la chaine de caracteres, si definie.
    const char *c_str( ) const
    {
    #ifndef GK_NAME_NOSTRING
        return m_name;
    #else
        return "(Name NOSTRING)";
    #endif
    }
    
    //! renvoie la chaine de caracteres, si definie.
    const char *c_str( )
    {
    #ifndef GK_NAME_NOSTRING
        return m_name;
    #else
        return "(Name NOSTRING)";
    #endif
    }
#endif
    
    //! comparaisons.
    friend bool operator==( const Name& a, const Name& b );
    //! comparaisons.
    friend bool operator!=( const Name& a, const Name& b );
    //! comparaisons.
    friend bool operator<( const Name& a, const Name& b );
    //! comparaisons.
    friend bool operator>( const Name& a, const Name& b );
    //! comparaisons.
    friend int compare( const Name& a, const Name& b );
    
    static
    unsigned int getHash( const char * string )
    {
        return murmur2(string, strlen(string), GK_NAME_HASH_SEED);
    }

    static
    unsigned int getHash( const std::string& string )
    {
        return murmur2(string.c_str(), string.size(), GK_NAME_HASH_SEED);
    }
    
    //! \todo construction incrementale a partir de plusieurs chaines / cles intermediaires, pour la construction des identifiants mesh, mesh_material, shader, etc ?
    //! cf murmurhash2a.cpp et ecrires les operateurs + << = += ? au lieu des fonctions utilisees dans l'exemple.
};

inline
bool operator==( const Name& a, const Name& b )
{
//~ #if 0
#ifndef GK_NAME_NOSTRING
    // verifie qu'il n'y pas de collisions
    const bool hash= (a.m_hash == b.m_hash);
    const int cmp= strcmp(a.m_name, b.m_name);
    assert((hash == true && cmp == 0) || (hash == false && cmp != 0));
#endif
    
    return (a.m_hash == b.m_hash);
}

inline
bool operator!=( const Name& a, const Name& b)
{
//~ #if 0
#ifndef GK_NAME_NOSTRING
    // verifie qu'il n'y pas de collisions
    const bool hash= (a.m_hash == b.m_hash);
    const int cmp= strcmp(a.m_name, b.m_name);
    assert((hash == true && cmp == 0) || (hash == false && cmp != 0));
#endif
    
    return (a.m_hash != b.m_hash);
}

inline
bool operator>( const Name& a, const Name& b )
{
//~ #if 0
#ifndef GK_NAME_NOSTRING
    // verifie qu'il n'y pas de collisions
    const bool hash= (a.m_hash == b.m_hash);
    const int cmp= strcmp(a.m_name, b.m_name);
    assert((hash == true && cmp == 0) || (hash == false && cmp != 0));
#endif
    
    return (a.m_hash > b.m_hash);
}

inline
bool operator<( const Name& a, const Name& b )
{
//~ #if 0
#ifndef GK_NAME_NOSTRING
    // verifie qu'il n'y pas de collisions
    const bool hash= (a.m_hash == b.m_hash);
    const int cmp= strcmp(a.m_name, b.m_name);
    assert((hash == true && cmp == 0) || (hash == false && cmp != 0));
#endif
    
    return (a.m_hash < b.m_hash);
}

inline
int compare( const Name& a, const Name& b )
{
//~ #if 0
#ifndef GK_NAME_NOSTRING
    // verifie qu'il n'y pas de collisions
    const bool hash= (a.m_hash == b.m_hash);
    const int cmp= strcmp(a.m_name, b.m_name);
    assert((hash == true && cmp == 0) || (hash == false && cmp != 0));
#endif

    if(a.m_hash < b.m_hash)
        return -1;
    else if(a.m_hash > b.m_hash)
        return 1;
    else
        return 0;
}

}

#endif

