
#include <cassert>

#include "GL/GLPlatform.h"
#include "Mesh.h"
#include "MeshIO.h"
#include "BufferManager.h"
#include "QueryManager.h"
#include "ShaderManager.h"

#include "App.h"


class TP : public gk::App
{
    gk::GLShaderProgram *m_program;
    gk::GLShaderProgram *m_feedback_program;
    gk::GLBuffer *m_positions;
    gk::GLBuffer *m_index;
    gk::GLBuffer *m_feedback_buffer;
    gk::GLQuery *m_feedback_query;
    int m_index_count;
    int m_position_count;
    
public:
    TP( const int w= 1024, const int h= 768 )
        :
        gk::App(w, h),
        m_program(NULL),
        m_positions(NULL),
        m_index(NULL),
        m_feedback_buffer(NULL),
        m_feedback_query(NULL),
        m_index_count(0),
        m_position_count(0)
    {}
    
    ~TP( ) {}
    
    int init( )         // charger les objets, les vertex buffers, etc.
    {
        // shader "classique" pour afficher un objet, utilise dans l'etape 2
        m_program= gk::createShaderProgram("core_simple.vsl", "core_simple.fsl");
        if(m_program == NULL || m_program->createGLResource() < 0)
            return -1;
        
        // charger un objet et creer les buffers
        gk::Mesh *mesh= gk::MeshIO::read("bigguy.obj");
        if(mesh == NULL)
            return -1;
        
        m_positions= gk::createBuffer(GL_ARRAY_BUFFER, mesh->positionCount() * sizeof(float [3]), &mesh->positions().front());
        if(m_positions == NULL || m_positions->createGLResource() < 0)
            return -1;
        m_position_count= mesh->positionCount();
        
        m_index= gk::createBuffer(GL_ELEMENT_ARRAY_BUFFER, mesh->indiceCount() * sizeof(int), &mesh->indices().front());
        if(m_index == NULL || m_index->createGLResource() < 0)
            return -1;
        m_index_count= mesh->indiceCount();
        
        // shader pour le feedback, utilise dans l'etape 1
        m_feedback_program= gk::createShaderProgram("core_feedback.vsl", "core_feedback.fsl");
        if(m_feedback_program == NULL)
            return -1;
        
        // configurer le feedback
        const char *varyings[]=
        {
            "vertex_position"
        };
        const int n= sizeof(varyings) / sizeof(const char *);
        printf("feedback: %d varyings\n", n);
        
        glTransformFeedbackVaryings(m_feedback_program->name(), n, varyings, GL_SEPARATE_ATTRIBS);
        assert(glGetError() == GL_NO_ERROR);
        
        // linker / creer le shader program
        if(m_feedback_program->createGLResource() < 0)
            return -1;
        
        // buffer pour le feedback, utiliser une taille suffisante ...
        m_feedback_buffer= gk::createBuffer(GL_ARRAY_BUFFER, 100000 * sizeof(float [4]), NULL, GL_DYNAMIC_COPY);
        if(m_feedback_buffer == NULL || m_feedback_buffer->createGLResource() < 0)
            return -1;
        
        // requete pour le feedback
        m_feedback_query= gk::createQuery(GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN);
        if(m_feedback_query == NULL || m_feedback_query->createGLResource() < 0)
            return -1;
        
        return 0;       // tout c'est bien passe
    }
    
    int quit( )         
    {
        // toutes les ressources seront liberees par shaderManager et bufferManager, etc.
        return 0;
    }
    
    int draw( )
    {
        if(key(SDLK_ESCAPE))
            // sortir si l'utilisateur appuye sur ESC
            Close();
        
        // redimensionner la fenetre, si necessaire.
        glViewport(0, 0, windowWidth(), windowHeight());
        // effacer le buffer de dessin.
        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        
        // transformations (communes aux 2 etapes)
        gk::Transform perspective= gk::Perspective(50.f, 1.f, 1.f, 1000.f);
        gk::Transform view= gk::Translate( gk::Vector(0.f, 0.f, -50.f) );
        gk::Transform model;    // identite
        // passer la matrice model view projection au shader
        gk::Transform mvp= perspective * view * model;
        
        // opengl core profile = shader necessaire.
        {
            // passe 1 : dessiner un objet et enregistrer les varyings du vertex shader dans un buffer
            glUseProgram(m_feedback_program->name());
            glUniformMatrix4fv( glGetUniformLocation(m_feedback_program->name(), "mvpMatrix"),
                1, GL_TRUE, mvp.matrix() );
            
            // parametrer les autres uniforms du shader
            glUniform4f( glGetUniformLocation(m_feedback_program->name(), "color"),
                1.f, .0f, .0f, 1.f);
            
            // activer les buffers d'attribut de sommets
            glBindBuffer(GL_ARRAY_BUFFER, m_positions->name());
            glVertexAttribPointer( glGetAttribLocation(m_feedback_program->name(), "position"), 
                3, GL_FLOAT, GL_FALSE, 0, NULL );
            // utiliser les buffers d'attributs...
            glEnableVertexAttribArray( glGetAttribLocation(m_feedback_program->name(), "position") );
            
            // activer les buffers du feedback
            glBindBufferBase(GL_TRANSFORM_FEEDBACK_BUFFER, 0, m_feedback_buffer->name());
            // compte le nombre d'attributs stockes dans le feedback
            m_feedback_query->begin();
            
            //~ glEnable(GL_RASTERIZER_DISCARD);        // ne pas dessiner les primitives ...
            // active le feedback
            glBeginTransformFeedback(GL_POINTS);
            {
                // traiter tous les sommets, et les "dessiner" sous forme de points.
                glDrawArrays(GL_POINTS, 0, m_position_count);
            }
            glEndTransformFeedback();
            m_feedback_query->end();
            //~ glDisable(GL_RASTERIZER_DISCARD);
            
            //~ const int count= m_feedback_query->result();
            //~ printf("feedback: %d primitives\n", count);
        }
        
        {
            // passe 2 : utiliser le buffer de feedback remplit par la premiere passe pour afficher un objet
            glUseProgram(m_program->name());
            glUniformMatrix4fv( glGetUniformLocation(m_program->name(), "mvpMatrix"),
                1, GL_TRUE, mvp.matrix() );
            
            // parametrer les autres uniforms du shader.
            glUniform4f( glGetUniformLocation(m_program->name(), "color"),
                .0f, 1.f, .0f, 1.f);    //en vert pour le feedback
            
            // utiliser le buffer de la passe 1 comme buffer d'attribut de sommet.
            glBindBuffer(GL_ARRAY_BUFFER, m_feedback_buffer->name());
            glVertexAttribPointer( glGetAttribLocation(m_program->name(), "position"), 
                4, GL_FLOAT, GL_FALSE, 0, NULL );
            // le varying enregistre dans le buffer est un vec4 !!
            // utiliser les buffers d'attributs...
            glEnableVertexAttribArray( glGetAttribLocation(m_program->name(), "position") );
            
            // activer le buffer d'indexation
            glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_index->name());
            // draw indexe, utilser le buffer d'indexation
            glDrawElements(GL_TRIANGLES, m_index_count, GL_UNSIGNED_INT, NULL);
        }
        
        // afficher le dessin.
        SDL_GL_SwapBuffers();
        
        // continuer.
        return 1;
    }
};


int main( int argc, char *argv[] )
{
    TP app;
    app.run();
    
    return 0;
}
