
#ifndef _TP_SHADER_PROGRAM_H
#define _TP_SHADER_PROGRAM_H

#include <string>
#include <vector>

#include "GLResource.h"
#include "GL/GLPlatform.h"
#include "GL/GLShaderObject.h"
#include "GL/GLShaderObjectIO.h"


namespace gk {

//! representation d'un shader program openGL.
class GLShaderProgram : public GLResource
{
protected:
    std::vector<GLShaderObject *> m_shaders;

    GLuint m_name;
    bool m_is_linked;
    bool m_is_validated;
    
    int make( );
    
public:
    //! constructeur par defaut.
    GLShaderProgram( )
        :
        m_shaders(),
        m_name(0),
        m_is_linked(false),
        m_is_validated(false)
    {
        printf("GLShaderProgram( )\n");
        m_name= glCreateProgram();
    }
    
    //! desctructeur.
    virtual ~GLShaderProgram( ) {}
    
    //! ajoute un shader object au shader program.
    int attachShader( GLShaderObject *shader );
    
    //! (re-)linke le shader program.
    int link( );
    //! valide la configuration du shader program.
    int validate( );
    
    //! construit le shader program. creation de l'objet openGL.
    int createGLResource( );
    
    //! detruit l'objet openGL.
    int releaseGLResource( );
    
    //! renvoie l'identifiant opengl de la ressource.
    GLuint name( ) const
    {
        return m_name;
    }
};

//! shader program openGL2, compose de :
//! - vertex shader,
//! - fragment shader.
class GL2ShaderProgram : public GLShaderProgram
{
public:
    GL2ShaderProgram( const std::string& vertex_filename, 
        const std::string& fragment_filename )
        :
        GLShaderProgram()
    {
        printf("GL2ShaderProgram( )\n");
        
        attachShader( GLVertexShaderIO::read(vertex_filename, "vertex") );
        attachShader( GLFragmentShaderIO::read(fragment_filename, "fragment") );
    }
    
    ~GL2ShaderProgram( ) {}
};


#if defined GK_OPENGL3 || defined GK_OPENGL4    
//! shader program openGL 3, compose de :
//! - vertex shader, 
//! - geometry shader,
//! - fragment shader.
//! definir GK_OPENGL3 ou GK_OPENGL4 lors de la compilation.
//! n'est utilisable que si la version de GLEW et le driver supportent GL3 ou GL4.
class GL3ShaderProgram : public GLShaderProgram
{
public:
    GL3ShaderProgram( const std::string& vertex_filename, 
        const std::string& geometry_filename, 
        const std::string& fragment_filename )
        :
        GLShaderProgram()
    {
        printf("GL3ShaderProgram( )\n");
        
        attachShader( GLVertexShaderIO::read(vertex_filename, "vertex") );
        attachShader( GLGeometryShaderIO::read(geometry_filename, "geometry") );
        attachShader( GLFragmentShaderIO::read(fragment_filename, "fragment") );
    }
    
    ~GL3ShaderProgram( ) {}
};
#endif

#ifdef GK_OPENGL4
//! shader program openGL 4, compose de :
//! - vertex shader, 
//! - tesselation control shader,
//! - tesselation evaluation shader,
//! - geometry shader,
//! - fragment shader.
//! definir GK_OPENGL4 lors de la compilation.
//! n'est utilisable que si la version de GLEW et le driver supportent GL4.
class GL4ShaderProgram : public GLShaderProgram
{
public:
    GL4ShaderProgram( const std::string& vertex_filename, 
        const std::string& control_filename, 
        const std::string& evaluation_filename, 
        const std::string& geometry_filename, 
        const std::string& fragment_filename )
        :
        GLShaderProgram()
    {
        printf("GL4ShaderProgram( )\n");
        
        attachShader( GLVertexShaderIO::read(vertex_filename, "vertex") );
        attachShader( GLControlShaderIO::read(control_filename, "control") );
        attachShader( GLEvaluationShaderIO::read(evaluation_filename, "evaluation") );
        attachShader( GLGeometryShaderIO::read(geometry_filename, "geometry") );
        attachShader( GLFragmentShaderIO::read(fragment_filename, "fragment") );
    }
    
    ~GL4ShaderProgram( ) {}
};
#endif

}

#endif
