
#ifndef _TP_BUFFER_H
#define _TP_BUFFER_H

#include "GLResource.h"
#include "GL/GLPlatform.h"


namespace gk {

class GLBuffer : public GLResource
{
    GLuint m_name;
    GLenum m_usage;
    unsigned int m_length;
    unsigned int m_count;
    
public:
    //! constructeur d'un buffer, un tableau de vecteurs.
    //! \param count nombre de vecteurs,
    //! \param length longueur en OCTETS du buffer,
    //! \param data pointeur sur les donnees a transferer dans le buffer, ou NULL pour initialiser un buffer vide,
    //! \param usage decrit le type d'utilisation du buffer.
    GLBuffer( const unsigned int count, const unsigned int length, const void *data, const GLenum usage= GL_STATIC_DRAW )
        :
        m_name(0),
        m_usage(usage),
        m_length(length),
        m_count(count)
    {
        glGenBuffers(1, &m_name);
        glBindBuffer(GL_ARRAY_BUFFER, m_name);
        glBufferData(GL_ARRAY_BUFFER, m_length, data, m_usage);
    }
    
    //! destructeur.
    ~GLBuffer( )
    {
        glDeleteBuffers(1, &m_name);
    }
    
    //! creation de la ressource openGL.
    int createGLResource( )
    {
        return (m_name != 0) ? 0 : -1;
    }
    
    //! destruction de la ressource openGL.
    int releaseGLResource( )
    {
        return (m_name != 0) ? 0 : -1;
    }

    //! renvoie le nom openGL (l'identifiant) du buffer.
    GLuint name( ) const
    {
        return m_name;
    }
    
    //! renvoie le nombre de vecteurs alloues (valeur passee au contructeur).
    unsigned int count( ) const
    {
        if(m_name == 0)
            return 0;
        return m_count;
    }
    
    //! efface le contenu du buffer.
    int clear( ) 
    {
        if(m_name == 0)
            return -1;
        
        glBindBuffer(GL_ARRAY_BUFFER, m_name);
        glBufferData(GL_ARRAY_BUFFER, m_length, NULL, m_usage);
        return 0;
    }
    
    //! modifie le contenu d'une partie du buffer.
    //! \param offset position du premier octet a modifier,
    //! \param length nombre d'octets a modifier.
    int update( const unsigned long int offset, const unsigned long int length, const void *data )
    {
        if(m_name == 0)
            return -1;
        if(offset + length > m_length)
            return -1;
        
        glBindBuffer(GL_ARRAY_BUFFER, m_name);
        glBufferSubData(GL_ARRAY_BUFFER, offset, length, data);
        return 0;
    }
    
    //! utilise le contenu du buffer comme attribut de sommets. interprete le contenu du buffer comme un tableau de vecteurs de dimension \param size.
    //! \param index identifiant de l'attribut declare dans le shader, cf. glGetAttribLocation(),
    //! \param size nombre de composantes des attributs, 3 pour une position, par exemple,
    //! \param type type opengl des composantes des attributs, GL_FLOAT, par exemple,
    //! \param stride lorsque plusieurs attributs sont entrelaces dans le meme buffer, indique le nombre d'octets occuppes par les attributs d'un sommet,
    //! \param offset position de l'attribut dans son groupe (en octet).
    /*! exemple : \n
        position et normal sont entrelaces dans le meme buffer :
           - position occupe les 3 premiers float, offset= 0
           - normal occuppe les 3 float suivants, offset= sizeof(float [3])
           - position et normal occuppent 6 floats en tout, stride= sizeof(float [6])
    \code
    buffer->bindAsVertexAttribute( 
        glGetAttribLocation(m_program, "position"), 
        3, GL_FLOAT, sizeof(float [6]), 0 );
        
    buffer->bindAsVertexAttribute( 
        glGetAttribLocation(m_program, "normal"), 
        3, GL_FLOAT, sizeof(float [6]), sizeof(float [3]) );
    \endcode
    */
    int bindAsVertexAttribute( const unsigned int index, 
        const int size, const GLenum type, 
        const unsigned long int stride= 0, const unsigned long int offset= 0 )
    {
        if(m_name == 0)
            return -1;
        
        glBindBuffer(GL_ARRAY_BUFFER, m_name);
        glVertexAttribPointer(index, size, type, GL_FALSE, stride, (const GLvoid *) offset);
        glVertexAttribDivisor(index, 0);
        glEnableVertexAttribArray(index);
        return 0;
    }
    
    //! utilise le contenu du buffer comme attribut d'instance. cf. bindAsVertexAttribute().
    //! \param divisor souvent egal a 1, pour que chaque instance obtienne un attribut different.
    int bindAsInstanceAttribute( const unsigned int index, 
        const int size, const GLenum type, 
        const unsigned long int stride= 0, const unsigned long int offset= 0,
        const int divisor= 1 )
    {
        if(m_name == 0)
            return -1;
        
        glBindBuffer(GL_ARRAY_BUFFER, m_name);
        glVertexAttribPointer(index, size, type, GL_FALSE, stride, (const GLvoid *) offset);
        glVertexAttribDivisor(index, divisor);
        glEnableVertexAttribArray(index);
    }
    
    //! utilise le contenu du buffer comme tableau d'index pour un affichage avec sommets partages. cf. glDrawElements().
    int bindAsIndex( )
    {
        if(m_name == 0)
            return -1;
        
        glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_name);
        return 0;
    }
};

}

#endif
