
#ifndef _GK_SCENE_H
#define _GK_SCENE_H

#include <vector>
#include <map>

#include "Name.h"
#include "Transform.h"

namespace gk {

//! classe de base d'un objet de la scene.
class ISceneObject
{
    Transform m_local_transform;
    ISceneObject *m_parent;
    
public:
    ISceneObject( )
        :
        m_parent( NULL )
    {}
    
    virtual ~ISceneObject( ) {}
    
    //! positionne et oriente l'objet.
    void setLocalTransform( const Transform& transform )
    {
        m_local_transform= transform;
    }
    
    //! renvoie la transformation associee a l'objet. fixee par setLocalTransform().
    const Transform &getLocalTransform( )
    {
        return m_local_transform;
    }
    
    //! positionne l'objet par rapport a un autre.
    //! la transformation donnee par setLocalTransform() devient relative a celle de l'objet parent.
    void setParent( ISceneObject *parent )
    {
        m_parent = parent;
    }
    
    //! calcule et renvoie la transformation de l'objet dans le repere de la scene.
    //! tient compte des transformations des parents, si necessaire.
    Transform getWorldTransform( );
    
    //! affiche l'objet.
    virtual int draw( const Transform& view )= 0;
};


//! representation d'une scene hierarchique.
//! la scene est composee d'un ensemble d'objets nommes.
//! la scene est proprietaire des objets, ils seront detruits par la scene.
class Scene
{
    std::vector<ISceneObject *> m_objects;
    typedef std::map<Name, int> names_map;
    names_map m_names;
    
public:
    //! construction d'une scene vide.
    Scene( ) {}
    //! detruit la scene et les objets associes.
    ~Scene( );
    
    //! insere un nouvel objet nomme dans la scene.
    //! renvoie -1 si le nom est deja utilise, ou 0 sinon.
    int insert( const Name& name, ISceneObject *object );
    
    //! renvoie le nombre d'objets de la scene.
    int count( ) const
    {
        return ( int ) m_objects.size();
    }
    
    //! renvoie un objet de la scene.
    ISceneObject *object( const int id ) const
    {
        assert( id >= 0 && id < ( int ) m_objects.size() );
        assert( m_objects[id] != NULL );
        
        return m_objects[id];
    }
    
    //! recherche un objet d'apres son nom.
    //! renvoie un pointeur sur l'objet s'il existe, ou NULL.
    ISceneObject *find( const Name& name ) const;
    
    //! affiche la scene.
    int draw( const Transform& view );
};

}

#endif
