/*
    animation d'un maillage par vertex shader
    
    TP 1 / M2PRO Images
    
    mailto:jciehl@bat710.univ-lyon1.fr
    janvier 2007
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include <math.h>
#include <time.h>

#include <GL/glew.h>
#include <GL/glu.h>
#include <SDL/SDL.h>

#include "sdlkit.h"
#include "glsl2.h"

#include "vec.h"
#include "model.h"
#include "maya_obj.h"

/* position et orientation du point de vue */
float camera_x, camera_y, camera_z;
float camera_angle, camera_angle2;
float camera_vitesse;

GLuint program;
GLint destination;
GLint dt;

float radians(float d)
{
    return d * 0.0174532925f;
}

void keyboard(void)
{
    SDLMod mod;
    unsigned char *keys;
    
    mod= SDL_GetModState();
    keys= sdlkit_get_keys();
    
    if(keys[SDLK_q] || keys[SDLK_ESCAPE])
        sdlkit_send_stop();

    // navigation
    if(keys[SDLK_UP])
    {
        if(mod&KMOD_CTRL || mod&KMOD_ALT)
        {
            // rotation de la camera sur X
            camera_angle2+= 1.f;
            if(camera_angle2 >= 360.f)
                    camera_angle2= 0.f;
        }
        else
        {
            // deplacement de la camera dans la direction de la visee
            camera_z-= cos(radians(camera_angle)) * camera_vitesse;
            camera_x-= sin(radians(camera_angle)) * camera_vitesse;
        }
    }
    
    if(keys[SDLK_DOWN])
    {
        if(mod&KMOD_CTRL || mod&KMOD_ALT)
        {
            // rotation de la camera sur Y+
            camera_angle2-= 1.f;
            if(camera_angle2 < 0.f)
                camera_angle2= 359.f;
        }
        else
        {
            // deplacement de la camera dans la direction opposee de la visee
            camera_z+= cos(radians(camera_angle)) * camera_vitesse;
            camera_x+= sin(radians(camera_angle)) * camera_vitesse;
        }
    }
    
    if(keys[SDLK_LEFT])
    {
        if(mod&KMOD_CTRL || mod&KMOD_ALT)
        {
            // rotation de la camera sur Y-
            camera_angle+= 1.f;
            if(camera_angle >= 360.f)
                camera_angle= 0.f;
        }
        else
        {
            // deplacement perpendiculaire a la direction de visee
            camera_z-= cos(radians(camera_angle + 90.f)) * camera_vitesse;
            camera_x-= sin(radians(camera_angle + 90.f)) * camera_vitesse;
        }
    }
    
    if(keys[SDLK_RIGHT])
    {
        if(mod&KMOD_CTRL || mod&KMOD_ALT)
        {
            // rotation de la camera sur Y+
            camera_angle-= 1.f;
            if(camera_angle < 0.f)
                camera_angle= 359.f;
        }
        else
        {
            // deplacement perpendiculaire a la direction de visee
            camera_z-= cos(radians(camera_angle - 90.f)) * camera_vitesse;
            camera_x-= sin(radians(camera_angle - 90.f)) * camera_vitesse;
        }
    }

    if(keys[SDLK_PAGEUP])
        camera_y+= camera_vitesse;
    
    if(keys[SDLK_PAGEDOWN])
        camera_y-= camera_vitesse;
}


// mise en place des parametres d'openGL
void initgl(void)
{
    glClearColor(0.0, 0.0, 0.0, 0.0);
    glClearDepth(1.);

    glShadeModel (GL_SMOOTH);
    
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_CULL_FACE);
    // glCullFace(GL_BACK);

    // glPolygonMode(GL_FRONT, GL_LINE);
    // glPolygonMode(GL_BACK, GL_LINE);

    camera_x= 0.f;
    camera_y= 0.f;
    camera_z= 50.f;
    camera_angle= 0.f;
    camera_vitesse= 10.f;
    
    glMatrixMode(GL_PROJECTION);	// selectionne la matrice de visualisation
    glLoadIdentity();				// reinitialise les transformations

    gluPerspective(50., 1., 1., 10000.);
}


/* affichage du modele 
 */
void model_display_debug(MODEL *model0, MODEL *model1, float t)
{
    FACE *face0;
    FACE *face1;
    int i, n;

    glColor3f(.5f, .5f, .5f);
    glLineWidth(1.);

    /* definit le temps utilise par l'interpolation de la position */
    glUniform1f(dt, t);
    
    glBegin(GL_TRIANGLES);
    n= model_get_faces_n(model0);
    for(i= 0; i < n; i++)
    {
        face0= model_get_face_ptr(model0, i);
        face1= model_get_face_ptr(model1, i);

        glNormal3fv(model_face_get_vertex_norm_ptr(model0, face0, 0));
        /* associe un parametre supplementaire a chaque sommet : sa position dans le futur */
        glVertexAttrib3fv(destination, model_face_get_vertex_ptr(model1, face1, 0));
        glVertex3fv(model_face_get_vertex_ptr(model0, face0, 0));

        glNormal3fv(model_face_get_vertex_norm_ptr(model0, face0, 1));
        glVertexAttrib3fv(destination, model_face_get_vertex_ptr(model1, face1, 1));
        glVertex3fv(model_face_get_vertex_ptr(model0, face0, 1));

        glNormal3fv(model_face_get_vertex_norm_ptr(model0, face0, 2));
        glVertexAttrib3fv(destination, model_face_get_vertex_ptr(model1, face1, 2));
        glVertex3fv(model_face_get_vertex_ptr(model0, face0, 2));
    }
    glEnd();
}



/* affichage */
void display(MODEL *model0, MODEL *model1, float t)
{
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    /* oriente la scene par rapport a la camera qui est restee en 0,0,0 */
    glRotatef(360.f - camera_angle2, 1.f, 0.f, 0.f);
    glRotatef(360.f - camera_angle, 0.f, 1.f, 0.f);
    glTranslatef(-camera_x, -camera_y, -camera_z);

    /* desssine la scene */
    model_display_debug(model0, model1, t);
    
    SDL_GL_SwapBuffers();
}


int main(int argc, char **argv)
{
    MODEL *model0;
    MODEL *model1;
    float t;
    
    srand48(time(NULL));

    if(argc != 3)
    {
        printf("usage: %s model_t0.obj model_t1.obj\n", argv[0]);
        return 0;
    }

    /* charge les modeles .obj */
    model0= NULL;
    if(model_is_obj(argv[1])==0)
        model_load_obj(&model0, argv[1]);

    model1= NULL;
    if(model_is_obj(argv[2])==0)
        model_load_obj(&model1, argv[2]);
    
    if(model0==NULL || model1==NULL)
    {
        printf("\n -- unable to load model\n");
        return 1;
    }
   
    /* trianguler et ajouter les normales aux modeles */
    model_set_triangles(&model0);
    model_set_vertex_norm(model0);
    model_set_triangles(&model1);
    model_set_vertex_norm(model1);
    
    if(model_get_faces_n(model0) != model_get_faces_n(model1))
    {
        printf("\n -- incompatible models\n");
        return 1;
    }
    
    /* creation de la fenetre / contexte openGL */
    sdlkit_init(600, 600);
 
    /* prepare l'utilisation des shaders*/
    glsl_init();
    
    /* charge les shaders et assemble le programme */
    program= glsl_program_init("animation.vsl", NULL);
    if(program==0)
        return -1;
    
    /* active le programme */
    glUseProgram(program);
    
    /* recupere les parametres */
    dt= glGetUniformLocation(program, "dt");
    destination= glGetAttribLocation(program, "destination");
    printf("params %d %d\n", dt, destination);
     
    /* visualisation interactive du modele */
    t= 0.0f;
    initgl();
    while(!sdlkit_stop())
    {
        sdlkit_events();
        keyboard();
        display(model0, model1, t);

        t+= 0.01f;
        if(t > 1.0f)
            t= 0.0f;

        sdlkit_anime();
    }
   
    model_free(model0);
    model_free(model1);

    glsl_quit();
    sdlkit_quit();
    printf("\ndone.\n");
    return 0;
}
