
#include <GL/glew.h>
#include <GL/gl.h>

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

/* 
    cf. http://http.download.nvidia.com/developer/SDK/Individual_Samples/DEMOS/OpenGL/simple_framebuffer_object.zip 

    http://www.nvidia.com/dev_content/nvopenglspecs/GL_EXT_framebuffer_object.txt
 */

#include "render_target.h"

#define VERBOSE 1
#define VERBOSE_DEBUG 0


static 
void CheckFramebufferStatus()
{
    GLenum status;
    
    status = (GLenum) glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT);
    
    switch(status) {
        case GL_FRAMEBUFFER_COMPLETE_EXT:
        break;
        
        case GL_FRAMEBUFFER_UNSUPPORTED_EXT:
            printf("Unsupported framebuffer format\n");
        break;
        
        case GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT_EXT:
            printf("Framebuffer incomplete, missing attachment\n");
        break;
        
    #if 0
        case GL_FRAMEBUFFER_INCOMPLETE_DUPLICATE_ATTACHMENT_EXT:
            printf("Framebuffer incomplete, duplicate attachment\n");
        break;
    #endif
        
        case GL_FRAMEBUFFER_INCOMPLETE_DIMENSIONS_EXT:
            printf("Framebuffer incomplete, attached images must have same dimensions\n");
        break;
        
        case GL_FRAMEBUFFER_INCOMPLETE_FORMATS_EXT:
            printf("Framebuffer incomplete, attached images must have same format\n");
        break;
        
        case GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER_EXT:
            printf("Framebuffer incomplete, missing draw buffer\n");
        break;
        
        case GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER_EXT:
            printf("Framebuffer incomplete, missing read buffer\n");
        break;
        
        default:
        break;
    }
}


RENDER_TARGET *render_target_create_format(int w, int h, GLenum texTarget, GLint texFormat, GLenum depthFormat)
{
    RENDER_TARGET *target= (RENDER_TARGET *) malloc(sizeof(RENDER_TARGET));
    assert(target != NULL);
    
    target->texWidth= w;
    target->texHeight= h;
    target->texInternalFormat= texFormat;
    target->texTarget= texTarget;
    target->filterMode= (target->texTarget == GL_TEXTURE_RECTANGLE_NV) ? GL_NEAREST : GL_LINEAR;
    target->maxCoordS= (target->texTarget == GL_TEXTURE_RECTANGLE_NV) ? target->texWidth : 1;
    target->maxCoordT= (target->texTarget == GL_TEXTURE_RECTANGLE_NV) ? target->texHeight : 1;

    // 
    glGenFramebuffersEXT(1, &target->fb);
    glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, target->fb);    

    // init texture (color renderbuffer)
    target->tex= 0;
    if(texFormat != 0)
    {
        glGenTextures(1, &target->tex);
        glBindTexture(target->texTarget, target->tex);
        glTexImage2D(target->texTarget, 0, target->texInternalFormat, target->texWidth, target->texHeight, 0, 
            GL_RGBA, GL_UNSIGNED_BYTE, NULL);
        
        glTexParameteri(target->texTarget, GL_TEXTURE_MIN_FILTER, target->filterMode);
        glTexParameteri(target->texTarget, GL_TEXTURE_MAG_FILTER, target->filterMode);
        glTexParameteri(target->texTarget, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
        glTexParameteri(target->texTarget, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
        
        glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, 
            GL_COLOR_ATTACHMENT0_EXT, target->texTarget, target->tex, 0);
    }
    
    CheckFramebufferStatus();
    
    target->depthTex= 0;
    if(depthFormat != 0)
    {
        glGenTextures(1, &target->depthTex);
        glBindTexture(target->texTarget, target->depthTex);
        glTexImage2D(target->texTarget, 0, depthFormat, target->texWidth, target->texHeight, 0, 
            GL_DEPTH_COMPONENT, GL_UNSIGNED_INT, NULL);
        
        glTexParameteri(target->texTarget, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
        glTexParameteri(target->texTarget, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
        glTexParameteri(target->texTarget, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
        glTexParameteri(target->texTarget, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
        glTexParameteri(target->texTarget, GL_DEPTH_TEXTURE_MODE, GL_INTENSITY);

        glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, 
            GL_DEPTH_ATTACHMENT_EXT, target->texTarget, target->depthTex, 0);
    }
    
    CheckFramebufferStatus();
    
    glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
    
    return target;
}

RENDER_TARGET *render_target_create(int w, int h)
{
    return render_target_create_format(w, h, 
        GL_TEXTURE_2D, GL_RGBA, 
        GL_DEPTH_COMPONENT);
}


int render_target_delete(RENDER_TARGET *target)
{
    if(target == NULL)
        return -1;

    if(target->tex != 0)
        glDeleteTextures(1, &target->tex);
    if(target->depthTex != 0)
        glDeleteTextures(1, &target->depthTex);
    glDeleteFramebuffersEXT(1, &target->fb);
    
    return 0;
}

int render_target_bind(RENDER_TARGET *target)
{
    if(target == NULL)
    {
        // restore default frame buffer
        glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
        
        return 0;
    }
    
    // bind render target
    glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, target->fb);
    
    return 0;
}

int render_target_disable(RENDER_TARGET *target)
{
    glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);
    
    return 0;
}



int render_target_bind_texture(RENDER_TARGET *target)
{
    if(target == NULL)
        return -1;
    if(target->tex == 0)
        return -1;

    glEnable(target->texTarget);  
    glBindTexture(target->texTarget, target->tex);

    return 0;
}

int render_target_bind_depth_texture(RENDER_TARGET *target)
{
    if(target == NULL)
        return -1;
    if(target->depthTex == 0)
        return -1;
    
    glEnable(GL_TEXTURE_2D);  
    glBindTexture(GL_TEXTURE_2D, target->depthTex);

    return 0;
}

int render_target_get_viewport(RENDER_TARGET *target, int viewport[4])
{
    if(target == NULL)
        return -1;
    
    viewport[0]= 0;
    viewport[1]= 0;
    viewport[2]= target->texWidth;
    viewport[3]= target->texHeight;
    
    return 0;
}


static void extFailed(char *message)
{
    printf("%-48s  [FAILED]\n", message);
    fflush(stdout);
}

static void extSupported(char *message)
{
#if VERBOSE
    printf("%-48s  [  OK  ]\n", message);
    fflush(stdout);
#endif
}


int render_target_init()
{
    GLenum err;

    err= glewInit();
    if(err != GLEW_OK)
    {
        printf("%s\n", glewGetErrorString(err));
        extFailed("GLEW");
        return -1;
    }
    
    if(!glewIsSupported("GL_VERSION_2_0"))
    {
        extFailed("OpenGL 2");
        return -1;
    }
    
    if(!glewIsSupported("GL_EXT_framebuffer_object"))
    {
        extFailed("framebuffer objects");
        return -1;
    }

#if VERBOSE_DEBUG
    extSupported("glew");
    extSupported("openGL 2");
#endif
#if VERBOSE
    extSupported("framebuffer objects");
#endif
    
    return 0;
}

int render_target_quit()
{
    return 0;
}

