/*
    shadercc shader.*v* shader.*f*
    
    mailto:jean-claude.iehl@liris.cnrs.fr
    fevrier 2008
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>

#include <GL/glew.h>
#include <GL/gl.h>

#include "sdlkit.h"
#include "glsl2.h"


static int get_shader_type(char *fname, GLenum* type)
{
    char *ext;
    
    if(fname == NULL)
        return -1;
    
    ext= strrchr(fname, '.');
    if(ext == NULL)
        return -1;
    
    if(strchr(ext, 'v') != NULL || strchr(ext, 'V') != NULL)
    {
        *type= GL_VERTEX_SHADER;
        return 0;
    }
    
    if(strchr(ext, 'f') != NULL || strchr(ext, 'F') != NULL)
    {
        *type= GL_FRAGMENT_SHADER;
        return 0;
    }
    
#ifndef NO_PROGRAM4
    if(strchr(ext, 'g') != NULL || strchr(ext, 'G') != NULL)
    {
        *type= GL_GEOMETRY_SHADER_EXT;
        return 0;
    }
#endif
    
    return -1;
}

int shader_compile(char *fname, GLenum type)
{
    GLchar *source= NULL;
    GLuint shader;
    GLint length;
    GLint compiled;
    
    if(glsl_load_source(&source, &length, fname) < 0 || source == NULL)
        return -1;
    
    shader= glCreateShader(type);
    assert(shader != 0);
    
#if 0
    const GLchar *sources[]= 
    {
        "#version 110\n",
        "#line 0\n",
        source
    };
    const GLsizei sources_n= sizeof(sources) / sizeof(GLchar *);

    glShaderSource(shader, sources_n, sources, NULL);

#else
    glShaderSource(shader, 1, (const GLchar **) &source, NULL);
#endif
    
    glCompileShader(shader);

    compiled= GL_FALSE;
    glGetShaderiv(shader, GL_COMPILE_STATUS, &compiled);
    if(compiled==GL_FALSE)
    {
        if(type==GL_VERTEX_SHADER)
            extFailed("vertex shader compilation");
        else if(type==GL_FRAGMENT_SHADER)
            extFailed("fragment shader compilation");
        else
            extFailed("shader? compilation");
        
        glsl_shader_source_info(shader, fname, source, type);
        free(source);
        return -1;
    }
    else
    {
        if(type==GL_VERTEX_SHADER)
            extSupported("vertex shader compilation");
        else if(type==GL_FRAGMENT_SHADER)
            extSupported("fragment shader compilation");
    }
    
    free(source);
    return 0;
}


int main(int argc, char **argv)
{
    GLenum shader_type;
    GLuint program;
    int i;
    
    if(argc < 2)
    {
        printf("usage: %s [-c] shader.*v* shader*f*\n", argv[0]);
        return 0;
    }
    
    sdlkit_init(64, 64);
    if(glsl_init() < 0)
        return 1;

    if(argc > 1 && strcmp(argv[1], "-c")==0)
    {
        // compiler separement tous les shaders de la ligne de commande
        for(i= 2; i < argc; i++)
        {
            if(get_shader_type(argv[i], &shader_type) < 0)
            {
                printf("  %s: unknown shader type.\n", argv[i]);
                return 1;
            }
            
            shader_compile(argv[i], shader_type);
        }
    }
    else if(argc > 1)
    {
        // compiler les 2 shaders et les linker
        char *vfname= NULL;
        char *ffname= NULL;

        // identifie le vertex shader et le fragment shader
        if(get_shader_type(argv[1], &shader_type) < 0)
        {
            printf("  %s: unknown shader type.\n", argv[i]);
            return 1;
        }
        if(shader_type == GL_VERTEX_SHADER)
            vfname= argv[1];
        else if(shader_type == GL_FRAGMENT_SHADER)
            ffname= argv[1];
        
        if(argc > 2)
        {
            if(get_shader_type(argv[2], &shader_type) < 0)
            {
                printf("  %s: unknown shader type.\n", argv[i]);
                return 1;
            }
            if(shader_type == GL_VERTEX_SHADER && vfname == NULL)
                vfname= argv[2];
            else if(shader_type == GL_FRAGMENT_SHADER && ffname == NULL)
                ffname= argv[2];
            else
            {
                printf("  shaders must be of different types.\n");
                return 1;
            }
        }
        
        // si un seul shader est present construire le programe en utilisant le pipeline fixe 
        program= glsl_program_init(vfname, ffname);
    }
    
    glsl_quit();
    sdlkit_quit();
    
    return 0;
}
