/*
    visu modeles
    
    mailto:jciehl@bat710.univ-lyon1.fr
    
    fevrier 2004
    fevrier 2006
    decembre 2006
*/

#ifdef _WIN32
#include <windows.h>
#else
#include <unistd.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include <string.h>
#include <time.h>
#include <math.h>

#include <GL/gl.h>

#include "model.h"
#include "model_clean.h"
#include "maya_obj.h"
#include "off.h"
#include "vec.h"
#include "view.h"


/* affichage du modele 
 */
void model_display_debug(MODEL *model)
{
    ATTR *iv;
    int i;

    glLineWidth(1.);
    for(i= 0; i < model->faces_n; i++)
    {
        if(model->faces[i].attr < 0)
            continue;
        
        iv= &model->attr[model->faces[i].attr];
        
        if(strcmp(model->mat[model->faces[i].mat].name, "diffuseLuminaire1SG")==0)
            // sources de lumieres en jaune
            glColor3f(.5f, .5f, 0.f);
        else
            // objets en blanc
            glColor3f(.5f, .5f, .5f);

        if(model->faces[i].n==3)
        {
            // glBegin(GL_LINE_LOOP);
            glBegin(GL_TRIANGLES);
            glNormal3fv(model->norm[model->faces[i].norm]);
            
            glVertex3fv(model->v[iv[0][gl_vertex]]);
            glVertex3fv(model->v[iv[1][gl_vertex]]);
            glVertex3fv(model->v[iv[2][gl_vertex]]);
            glEnd();
        }
        else if(model->faces[i].n==4)
        {
            //~ printf("quad %d (attr %d)\n", i, model->faces[i].attr);
            
            glColor3f(1.f, 0.f, 0.f);
            // glBegin(GL_LINE_LOOP);
            glBegin(GL_QUADS);
            glNormal3fv(model->norm[model->faces[i].norm]);

            glVertex3fv(model->v[iv[0][gl_vertex]]);
            glVertex3fv(model->v[iv[1][gl_vertex]]);
            glVertex3fv(model->v[iv[2][gl_vertex]]);
            glVertex3fv(model->v[iv[3][gl_vertex]]);
            glEnd();
        }
    }
}


void model_display_init(MODEL *model)
{
    int i;
    
    if(model->draw_index!=NULL)
        return;
    
    // TODO : check that vbo_clean is set
    
    model->draw_index= malloc(sizeof(int) * model->attr_n);
    assert(model->draw_index!=NULL);
    
    for(i= 0; i < model->attr_n; i++)
        model->draw_index[i]= model->attr[i][gl_vertex];
    
    model->draw_index_n= model->attr_n;
}	

void model_display(MODEL *model)
{
    glLineWidth(1.f);

    glEnableClientState(GL_VERTEX_ARRAY);
    glVertexPointer(3, GL_FLOAT, sizeof(VERTEX), model->v);
    
    if((model->flags & MODEL_HAS_NORM)!=0)
    {
        glEnableClientState(GL_NORMAL_ARRAY);
        glNormalPointer(GL_FLOAT, sizeof(VERTEXN), model->norm);
    }
    else
        glDisableClientState(GL_NORMAL_ARRAY);
    
    glDrawElements(GL_TRIANGLES, model->draw_index_n, 
        GL_UNSIGNED_INT, model->draw_index);
}


int main(int argc, char **argv)
{
    char *clean_fname= NULL;
    char *fname= NULL;
    MODEL *model;
    int clean, vbo;
    
    srand(time(NULL));

    view_init(&argc, &argv);

    if(argc < 2 || argc > 4)
    {
        printf("usage: %s [--vbo output.obj] filename.obj\n", argv[0]);
        printf("usage: %s [--save output.obj] filename.obj\n", argv[0]);
        return 0;
    }

    //
    clean= 0;
    vbo= 0;
    if(argc==2)
        fname= argv[1];
    
    else if(argc > 2) 
    {
        if(strcmp(argv[1], "--save")==0)
            clean= 1;
        else if(strcmp(argv[1], "--vbo")==0)
            vbo= 1;
        else
        {
            printf("usage: %s [--vbo output.obj] filename.obj\n", argv[0]);
            printf("usage: %s [--save output.obj] filename.obj\n", argv[0]);
            return 1;
        }
        
        if(argc==3)
            fname= argv[2];
        else
        {
            clean_fname= argv[2];
            fname= argv[3];
        }
    }

    if(fname==NULL)
    {
        printf("usage: %s [--vbo output.obj] filename.obj\n", argv[0]);
        printf("usage: %s [--save output.obj] filename.obj\n", argv[0]);
        return 1;
    }
    
    model= NULL;
    if((model_is_obj(fname)==0 && model_load_obj(&model, fname) < 0)
    || (model_is_off(fname)==0 && model_load_off(&model, fname) < 0))
    {
        printf("\n  -- unable to load model '%s'\n", fname);
        return 1;
    }
    
    if(clean!=0)
    {
        printf("model '%s' : forced clean\n", fname);
        model_clean(&model);
    }
    if(vbo!=0)
    {
        printf("model '%s' : forced vbo clean\n", fname);
        model_vbo_clean(&model);
    }
    
    if(clean_fname!=NULL)
    {
        if(model_save_obj(model, clean_fname) < 0)
        {
            printf("\n  -- unable to save model '%s'\n", clean_fname);
            return 1;
        }

        printf("\nsave model '%s'\n", clean_fname);
    }
    
    if(model_is_vbo_clean(model)==0)
    {
        printf("model '%s' : forced vbo clean\n", fname);
        model_vbo_clean(&model);
    }

#if 0
    /* teste la subdivision
     */
    {
        float area= model_get_area(model);
        printf("model area %f\n", area);
        model_subdivide(&model, area / 10000.f);
        clean= 1;       // force clean
    }
#endif
    
    model_set_faces_norm(model);
    
    model_display_init(model);
    view((void *) model_display_debug, (void *) model, model->filename);

    model_free(model);
    return 0;
}
