#ifndef _MODEL_PRIVATE_H
#define _MODEL_PRIVATE_H

#include <assert.h>
#include "model.h"

static inline 
int model_get_faces_n(const MODEL *model)
{
    assert(model!=NULL);
    
    return model->faces_n;
}

static inline 
int model_get_vertex_n(const MODEL *model)
{
    assert(model!=NULL);

    return model->v_n;
}

static inline
int model_get_texcoord_n(const MODEL *model)
{
    assert(model != NULL);
    
    return model->tex_n;
}

static inline 
FACE *model_get_face_ptr(const MODEL *model, const int f)
{
    assert(model!=NULL);
    assert(f>=0 && f < model->faces_n);
    
    return &model->faces[f];
}

static inline 
int face_get_vertex_n(const FACE *face)
{
    assert(face!=NULL);
    
    return face->n;
}

static inline 
int face_get_attr_id(const FACE *face, const int a)
{
    assert(face!=NULL);
    assert(a >= 0 && a < face->n);

    return face->attr + a;
}

static inline 
int face_get_material_id(const FACE *face)
{
    assert(face!=NULL);

    return face->mat;
}

static inline 
int face_get_norm_id(const FACE *face)
{
    assert(face!=NULL);
    
    return face->norm;
}

static inline
int model_attr_id_set(MODEL *model, const int a, const int vertex_id, const int texcoord_id, const int norm_id)
{
    assert(model != NULL);
    assert(a >= 0 && a < model->attr_n);
    
    model->attr[a][gl_vertex]= vertex_id;
    model->attr[a][gl_tex]= texcoord_id;
    model->attr[a][gl_norm]= norm_id;
    
    return 0;
}

static inline
ATTR *model_get_attr_ptr(const MODEL *model, const int a)
{
    assert(model != NULL);
    assert(a >= 0 && a < model->attr_n);
    
    return &model->attr[a];
}

static inline 
int model_attr_get_vertex_id(const MODEL *model, const int a)
{
    assert(model!=NULL);
    assert(a>=0 && a < model->attr_n);
    
    return model->attr[a][gl_vertex];
}

static inline 
float *model_attr_get_vertex_ptr(const MODEL *model, const int a)
{
    assert(model!=NULL);
    assert(a>=0 && a < model->attr_n);
    
    return model->v[model->attr[a][gl_vertex]];
}

static inline 
int model_attr_get_norm_id(const MODEL *model, const int a)
{
    assert(model!=NULL);
    assert(a>=0 && a < model->attr_n);
    
    return model->attr[a][gl_norm];
}

static inline 
float *model_attr_get_norm_ptr(const MODEL *model, const int a)
{
    assert(model!=NULL);
    assert(a>=0 && a < model->attr_n);
    assert(model->attr[a][gl_norm] >= 0);
    
    return model->norm[model->attr[a][gl_norm]];
}

static inline 
int model_attr_get_texcoord_id(const MODEL *model, const int a)
{
    assert(model!=NULL);
    assert(a>=0 && a < model->attr_n);
    
    return model->attr[a][gl_tex];
}

static inline 
float *model_attr_get_texcoord_ptr(const MODEL *model, const int a)
{
    assert(model!=NULL);
    assert(a>=0 && a < model->attr_n);
    
    return model->tex[model->attr[a][gl_tex]];
}


static inline 
float *model_get_vertex_ptr(const MODEL *model, const int v)
{
    assert(model != NULL);
    assert(v >= 0 && v < model->v_n);
    
    return model->v[v];
}

static inline 
float *model_get_texcoord_ptr(const MODEL *model, const int t)
{
    assert(model != NULL);
    assert(t >= 0 && t < model->tex_n);
    
    return model->tex[t];
}

static inline 
float *model_get_norm_ptr(const MODEL *model, const int n)
{
    assert(model!=NULL);
    assert(n>=0 && n<model->norm_n);
    
    return model->norm[n];
}

static inline 
MATERIAL *model_get_material_ptr(const MODEL *model, const int m)
{
    assert(model!=NULL);
    assert(m>=0 && m < model->mat_n);
    
    return &model->mat[m];
}


static inline 
MATERIAL *model_face_get_material_ptr(const MODEL *model, const FACE *face)
{
    return model_get_material_ptr(model, face_get_material_id(face));
}

static inline 
float *model_face_get_norm_ptr(const MODEL *model, const FACE *face)
{
    return model_get_norm_ptr(model, face_get_norm_id(face));
}

static inline 
int model_face_get_vertex_id(const MODEL *model, const FACE *face, const int v)
{
    assert(model!=NULL);
    assert(face!=NULL);
    assert(v>=0 && v < face->n);
    
    const int attr= face->attr + v;
    return model_attr_get_vertex_id(model, attr);
}

static inline 
float *model_face_get_vertex_ptr(const MODEL *model, const FACE *face, const int v)
{
    assert(model!=NULL);
    assert(face!=NULL);
    assert(v>=0 && v < face->n);
    
    const int attr= face->attr + v;
    return model_attr_get_vertex_ptr(model, attr);
}


static inline 
int model_face_get_vertex_norm_id(const MODEL *model, const FACE *face, const int v)
{
    assert(model != NULL);
    assert(face != NULL);
    assert(v >= 0 && v < face->n);
    
    const int attr= face->attr + v;
    return model_attr_get_norm_id(model, attr);
}

static inline 
float *model_face_get_vertex_norm_ptr(const MODEL *model, const FACE *face, const int v)
{
    assert(model!=NULL);
    assert(face!=NULL);
    assert(v>=0 && v < face->n);
    
    const int attr= face->attr + v;
    return model_attr_get_norm_ptr(model, attr);
}

static inline 
int model_face_get_vertex_texcoord_id(const MODEL *model, const FACE *face, const int v)
{
    assert(model != NULL);
    assert(face != NULL);
    assert(v >= 0 && v < face->n);
    
    const int attr= face->attr + v;
    return model_attr_get_texcoord_id(model, attr);
}

static inline 
float *model_face_get_vertex_texcoord_ptr(const MODEL *model, const FACE *face, const int v)
{
    assert(model!=NULL);
    assert(face!=NULL);
    assert(v>=0 && v < face->n);
    
    const int attr= face->attr + v;
    return model_attr_get_texcoord_ptr(model, attr);
}

static inline 
int model_face_get_edge(const MODEL *model, const FACE *face, const int e, VEC *edge)
{
    const float *v0= model_face_get_vertex_ptr(model, face, e);
    const float *v1= model_face_get_vertex_ptr(model, face, (e+1) % face->n);
    
    vec3_sub(*edge, v1, v0);
    return 0;
}

static inline 
int model_face_get_inner_edge(const MODEL *model, const FACE *face, const int e, VEC *edge)
{
    const float *v0= model_face_get_vertex_ptr(model, face, 0);
    const float *v1= model_face_get_vertex_ptr(model, face, e);
    
    vec3_sub(*edge, v1, v0);
    return 0;
}

#endif
