/*
    vertex_buffer_testmain.c
    
    tests elementaires vertex_buffer.c
    exemples d'utilisation
    
    mailto:jean-claude.iehl@liris.cnrs.fr
    fevrier 2008
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <GL/glew.h>
#include <GL/gl.h>


#include "vec.h"
#include "model.h"
#include "model_clean.h"
#include "maya_obj.h"
#include "off.h"
#include "sdlkit.h"

#include "vertex_buffer.h"


#define checkGLerror(a) { \
    GLenum code= glGetError(); \
    if(code != GL_NO_ERROR) \
        printf("[%s line %d] GL Error: %s\n",  __FILE__, __LINE__, \
            gluErrorString(code)); }

static char *object_fname= "bigguy.obj";
static MODEL *object;
static VERTEX_BUFFER *object_buffer;

static char *ground_fname= "ground.obj";
static MODEL *ground;
static VERTEX_BUFFER *ground_buffer;

// position et orientation du point de vue
static float camera_x, camera_y, camera_z;
static float camera_angle, camera_angle2;
static float camera_vitesse;

static int lumiere;

// conversion degres vers radians
float radians(float d)
{
    return d * 0.0174532925f;
}

void keyboard(void)
{
    SDLMod mod;
    unsigned char *keys;
    
    mod= SDL_GetModState();
    keys= sdlkit_get_keys();
    
    if(keys[SDLK_q] || keys[SDLK_ESCAPE])
        sdlkit_send_stop();

    if(keys[SDLK_SPACE])
        lumiere= (lumiere +1) %2;

    // navigation
    if(keys[SDLK_UP])
    {
        if(mod&KMOD_CTRL || mod&KMOD_ALT)
        {
            // rotation de la camera sur X
            camera_angle2+= 1.f;
            if(camera_angle2 >= 360.f)
                    camera_angle2= 0.f;
        }
        else
        {
            // deplacement de la camera dans la direction de la visee
            camera_z-= cos(radians(camera_angle)) * camera_vitesse;
            camera_x-= sin(radians(camera_angle)) * camera_vitesse;
        }
    }
    
    if(keys[SDLK_DOWN])
    {
        if(mod&KMOD_CTRL || mod&KMOD_ALT)
        {
            // rotation de la camera sur Y+
            camera_angle2-= 1.f;
            if(camera_angle2 < 0.f)
                camera_angle2= 359.f;
        }
        else
        {
            // deplacement de la camera dans la direction opposee de la visee
            camera_z+= cos(radians(camera_angle)) * camera_vitesse;
            camera_x+= sin(radians(camera_angle)) * camera_vitesse;
        }
    }
    
    if(keys[SDLK_LEFT])
    {
        if(mod&KMOD_CTRL || mod&KMOD_ALT)
        {
            // rotation de la camera sur Y-
            camera_angle+= 1.f;
            if(camera_angle >= 360.f)
                camera_angle= 0.f;
        }
        else
        {
            // deplacement perpendiculaire a la direction de visee
            camera_z-= cos(radians(camera_angle + 90.f)) * camera_vitesse;
            camera_x-= sin(radians(camera_angle + 90.f)) * camera_vitesse;
        }
    }
    
    if(keys[SDLK_RIGHT])
    {
        if(mod&KMOD_CTRL || mod&KMOD_ALT)
        {
            // rotation de la camera sur Y+
            camera_angle-= 1.f;
            if(camera_angle < 0.f)
                camera_angle= 359.f;
        }
        else
        {
            // deplacement perpendiculaire a la direction de visee
            camera_z-= cos(radians(camera_angle - 90.f)) * camera_vitesse;
            camera_x-= sin(radians(camera_angle - 90.f)) * camera_vitesse;
        }
    }

    if(keys[SDLK_PAGEUP])
        camera_y+= camera_vitesse;
    
    if(keys[SDLK_PAGEDOWN])
        camera_y-= camera_vitesse;
}

static int model_set_vertex_buffer(MODEL *model, VERTEX_BUFFER *buffer)
{
    if(vertex_buffer_add_attribute(buffer, VERTEX_ATTR, 3, GL_FLOAT, 
        model->v_n, sizeof(VERTEX), model->v) < 0)
        printf("add vertex boom\n");
    
    checkGLerror();    

    if(vertex_buffer_add_attribute(buffer, NORMAL_ATTR, 3, GL_FLOAT, 
        model->norm_n, sizeof(VERTEXN), model->norm) < 0)
        printf("add normal boom\n");

    checkGLerror();    
    
    if(vertex_buffer_add_index(buffer, GL_UNSIGNED_INT, 
        model->attr_n, sizeof(ATTR), &model->attr[0][gl_vertex]) < 0)
        printf("add index boom\n");

    checkGLerror();    
    
    return 0;
}


static int load_scene(void)
{
    object= NULL;
    if(model_is_obj(object_fname)==0)
        model_load_obj(&object, object_fname);
    if(model_is_off(object_fname)==0)
        model_load_off(&object, object_fname);
    if(object==NULL)
    {
        printf("\n -- unable to load model '%s'\n", object_fname);
        return -1;
    }

    model_set_triangles(&object);
    model_set_vertex_norm(object);
    model_vbo_clean(&object);
    object_buffer= vertex_buffer_create(GL_TRIANGLES);
    checkGLerror();
    assert(object_buffer != NULL);
    model_set_vertex_buffer(object, object_buffer);
    checkGLerror();    
    
    //
    ground= NULL;
    if(model_is_obj(ground_fname)==0)
        model_load_obj(&ground, ground_fname);
    if(model_is_off(ground_fname)==0)
        model_load_off(&ground, ground_fname);
    if(ground==NULL)
    {
        printf("\n -- unable to load model '%s'\n", ground_fname);
        return -1;
    }

    model_set_triangles(&ground);
    model_set_vertex_norm(ground);
    model_vbo_clean(&ground);
    ground_buffer= vertex_buffer_create(GL_TRIANGLES);
    checkGLerror();
    assert(ground_buffer != NULL);
    model_set_vertex_buffer(ground, ground_buffer);
    checkGLerror();
    
    return 0;
}

// mise en place des parametres d'openGL
void initgl(void)
{
    glClearColor(0.0, 0.0, 0.0, 0.0);
    glClearDepth(1.);

    glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST);
    glShadeModel (GL_SMOOTH);
    
    glEnable(GL_DEPTH_TEST);
    glDepthFunc(GL_LEQUAL);
    glEnable(GL_CULL_FACE);
    glCullFace(GL_BACK);
    
    glPolygonMode(GL_FRONT, GL_FILL);
    glPolygonMode(GL_BACK, GL_FILL);
    
    camera_x= 0.f;
    camera_y= 0.f;
    camera_z= 50.f;
    camera_angle= 0.f;
    camera_vitesse= 4.f;
    
    lumiere= 0;
}

int model_display_vbo(MODEL *model, VERTEX_BUFFER *buffer)
{
    glColor3f(1.f, .5f, .5f);
    if(vertex_buffer_check(buffer) < 0)
        printf("vertex buffer check BOOM\n");
    if(vertex_buffer_draw(buffer) < 0)
        printf("boom!!\n");
    checkGLerror();    
    
    return 0;
}


int main(void)
{
    
    /* creation de la fenetre / contexte openGL */
    sdlkit_init(600, 600);
   
    if(vertex_buffer_init() < 0)
        goto done;
    
    if(load_scene() < 0)
        goto done;
   
    /* fixe l'etat par defaut */
    initgl();

    while(!sdlkit_stop())
    {
        sdlkit_events();
        keyboard();
    
        int viewport[4];
        sdlkit_get_viewport(viewport);
        glViewport(viewport[0], viewport[1], viewport[2], viewport[3]);

        glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

        glMatrixMode(GL_PROJECTION);
        glLoadIdentity();
        gluPerspective(50., 1., 1., 1000.);
        
        glMatrixMode(GL_MODELVIEW);
        glLoadIdentity();

        // oriente la scene par rapport a la camera qui est restee en 0,0,0
        glRotatef(360.f - camera_angle2, 1.f, 0.f, 0.f);
        glRotatef(360.f - camera_angle, 0.f, 1.f, 0.f);
        glTranslatef(-camera_x, -camera_y, -camera_z);
        
        // fixe une lumiere par defaut, si necessaire
        if(lumiere!=0)
        {
            // active une source de lumiere et definit une matiere par defaut
            GLfloat mat_specular[] = { 1.0f, 1.0f, 1.0f, 1.0f };
            GLfloat mat_shininess[] = { 20.0f };
            GLfloat light_position[] = { 0.0f, 1000.0f, 100.0f, 0.0f };
            
            glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
            glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);
            glLightfv(GL_LIGHT0, GL_POSITION, light_position);
            
            glEnable(GL_LIGHTING);
            glEnable(GL_LIGHT0);
            glPolygonMode(GL_FRONT, GL_FILL);
            glPolygonMode(GL_BACK, GL_FILL);
        }
        else
        {
            // desactive la source de lumiere
            glDisable(GL_LIGHTING);
            glPolygonMode(GL_FRONT, GL_LINE);
            glPolygonMode(GL_BACK, GL_LINE);
        }
    
        // ... et dessine la scene
        model_display_vbo(object, object_buffer);
        model_display_vbo(ground, ground_buffer);
    
        SDL_GL_SwapBuffers();
        sdlkit_anime();
    }

    vertex_buffer_delete(object_buffer);
    vertex_buffer_delete(ground_buffer);
    vertex_buffer_quit();
    
done:    
    printf("\ndone.\n");
    sdlkit_quit();
    return 0;
}
