/*
    animation d'un maillage par vertex shader
    
    exemple d'utilisation des vertex_buffer et d'un vertex shader
    
    mailto:jciehl@bat710.univ-lyon1.fr
    fevrier 2008
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include <math.h>
#include <time.h>

#include <GL/glew.h>
#include <GL/gl.h>
#include <SDL/SDL.h>

#include "sdlkit.h"
#include "glsl2.h"
#include "vertex_buffer.h"

#include "vec.h"
#include "model.h"
#include "model_clean.h"
#include "maya_obj.h"

/* position et orientation du point de vue */
float camera_x, camera_y, camera_z;
float camera_angle, camera_angle2;
float camera_vitesse;

GLuint program;
GLint destination_location;
GLint dt_location;

float radians(float d)
{
    return d * 0.0174532925f;
}

void keyboard(void)
{
    SDLMod mod;
    unsigned char *keys;
    
    mod= SDL_GetModState();
    keys= sdlkit_get_keys();
    
    if(keys[SDLK_q] || keys[SDLK_ESCAPE])
        sdlkit_send_stop();

    // navigation
    if(keys[SDLK_UP])
    {
        if(mod&KMOD_CTRL || mod&KMOD_ALT)
        {
            // rotation de la camera sur X
            camera_angle2+= 1.f;
            if(camera_angle2 >= 360.f)
                    camera_angle2= 0.f;
        }
        else
        {
            // deplacement de la camera dans la direction de la visee
            camera_z-= cos(radians(camera_angle)) * camera_vitesse;
            camera_x-= sin(radians(camera_angle)) * camera_vitesse;
        }
    }
    
    if(keys[SDLK_DOWN])
    {
        if(mod&KMOD_CTRL || mod&KMOD_ALT)
        {
            // rotation de la camera sur Y+
            camera_angle2-= 1.f;
            if(camera_angle2 < 0.f)
                camera_angle2= 359.f;
        }
        else
        {
            // deplacement de la camera dans la direction opposee de la visee
            camera_z+= cos(radians(camera_angle)) * camera_vitesse;
            camera_x+= sin(radians(camera_angle)) * camera_vitesse;
        }
    }
    
    if(keys[SDLK_LEFT])
    {
        if(mod&KMOD_CTRL || mod&KMOD_ALT)
        {
            // rotation de la camera sur Y-
            camera_angle+= 1.f;
            if(camera_angle >= 360.f)
                camera_angle= 0.f;
        }
        else
        {
            // deplacement perpendiculaire a la direction de visee
            camera_z-= cos(radians(camera_angle + 90.f)) * camera_vitesse;
            camera_x-= sin(radians(camera_angle + 90.f)) * camera_vitesse;
        }
    }
    
    if(keys[SDLK_RIGHT])
    {
        if(mod&KMOD_CTRL || mod&KMOD_ALT)
        {
            // rotation de la camera sur Y+
            camera_angle-= 1.f;
            if(camera_angle < 0.f)
                camera_angle= 359.f;
        }
        else
        {
            // deplacement perpendiculaire a la direction de visee
            camera_z-= cos(radians(camera_angle - 90.f)) * camera_vitesse;
            camera_x-= sin(radians(camera_angle - 90.f)) * camera_vitesse;
        }
    }

    if(keys[SDLK_PAGEUP])
        camera_y+= camera_vitesse;
    
    if(keys[SDLK_PAGEDOWN])
        camera_y-= camera_vitesse;
}


// mise en place des parametres d'openGL
void initgl(void)
{
    glClearColor(0.0, 0.0, 0.0, 0.0);
    glClearDepth(1.);

    glShadeModel (GL_SMOOTH);
    
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_CULL_FACE);
    glCullFace(GL_BACK);

    // glPolygonMode(GL_FRONT, GL_LINE);
    // glPolygonMode(GL_BACK, GL_LINE);

    camera_x= 0.f;
    camera_y= 0.f;
    camera_z= 50.f;
    camera_angle= 0.f;
    camera_vitesse= 10.f;
    
    glMatrixMode(GL_PROJECTION);	// selectionne la matrice de visualisation
    glLoadIdentity();				// reinitialise les transformations

    gluPerspective(50., 1., 1., 10000.);
}


/* affichage */
void display(VERTEX_BUFFER *buffer, float t)
{
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    /* oriente la scene par rapport a la camera qui est restee en 0,0,0 */
    glRotatef(360.f - camera_angle2, 1.f, 0.f, 0.f);
    glRotatef(360.f - camera_angle, 0.f, 1.f, 0.f);
    glTranslatef(-camera_x, -camera_y, -camera_z);

    /* desssine la scene 
     */
    
    // modifie le parametre du shader d'animation
    glUniform1f(dt_location, t);

    // affiche le modele avec le vertex buffer
    vertex_buffer_draw(buffer);
    
    SDL_GL_SwapBuffers();
}


int main(int argc, char **argv)
{
    MODEL *model0;
    MODEL *model1;
    VERTEX_BUFFER *buffer0;
    float t;
    
    srand48(time(NULL));

    if(argc != 3)
    {
        printf("usage: %s model_t0.obj model_t1.obj\n", argv[0]);
        return 0;
    }

    /* creation de la fenetre / contexte openGL */
    sdlkit_init(600, 600);

    /* prepare l'utilisation des vertex buffers */
    if(vertex_buffer_init() < 0)
        return 0;

    /* charge les modeles .obj */
    model0= NULL;
    if(model_is_obj(argv[1])==0)
        model_load_obj(&model0, argv[1]);

    model1= NULL;
    if(model_is_obj(argv[2])==0)
        model_load_obj(&model1, argv[2]);
    
    if(model0==NULL || model1==NULL)
    {
        printf("\n -- unable to load model\n");
        return 1;
    }
   
    /* trianguler et ajouter les normales aux modeles */
    model_set_triangles(&model0);
    model_set_vertex_norm(model0);
    /* preparer le modele pour l'affichage avec un vertex_buffer */
    model_vbo_clean(&model0);
    
    model_set_triangles(&model1);
    model_set_vertex_norm(model1);
    /* preparer le modele pour l'affichage avec un vertex_buffer */
    model_vbo_clean(&model1);

    /* quelques tests */
    if(model_get_faces_n(model0) != model_get_faces_n(model1))
    {
        printf("\n -- incompatible models\n");
        return 1;
    }
    
    /* creation du vertex buffer */
    buffer0= vertex_buffer_create(GL_TRIANGLES);

    /* remplit le buffer avec les sommets, les normales et les indices des sommets */
    vertex_buffer_add_attribute(buffer0, VERTEX_ATTR, 3, GL_FLOAT, 
        model0->v_n, sizeof(VERTEX), model0->v);
    
    vertex_buffer_add_attribute(buffer0, NORMAL_ATTR, 3, GL_FLOAT, 
        model0->norm_n, sizeof(VERTEXN), model0->norm);
    
    vertex_buffer_add_index(buffer0, GL_UNSIGNED_INT, 
        model0->attr_n, sizeof(ATTR), &model0->attr[0][gl_vertex]);
    // TODO : a mettre dans model.c ...
    
    if(vertex_buffer_check(buffer0))
    {
        printf("vbo check 1, FAILED.\n");
    }

    
    /* prepare l'utilisation des shaders*/
    if(glsl_init() < 0)
        return 0;
    
    /* charge les shaders et linke le programme */
    program= glsl_program_init("animation.vsl", NULL);
    if(program==0)
        return 0;
    
    /* active le programme */
    glUseProgram(program);
    
    /* recupere les parametres */
    dt_location= glGetUniformLocation(program, "dt");
    destination_location= glGetAttribLocation(program, "destination");

    /* utilise les positions des sommets du model1 comme attributs des sommets du model0 (pour le shader) */
    vertex_buffer_add_attribute(buffer0, VERTEXATTR0_ATTR, 3, GL_FLOAT,
        model1->v_n, sizeof(VERTEX), model1->v);
    
    /* associe les sommets de model1 au parametre destination du shader */
    vertex_buffer_bind_attribute_location(buffer0, VERTEXATTR0_ATTR, destination_location);
    
    if(vertex_buffer_check(buffer0))
    {
        printf("vbo check 2, FAILED.\n");
    }
    
    
    /* affiche l'animation du modele */
    t= 0.0f;
    initgl();
    while(!sdlkit_stop())
    {
        sdlkit_events();
        keyboard();
        display(buffer0, t);
        
        t+= 0.01f;
        if(t > 1.0f)
            t= 0.0f;

        sdlkit_anime();
    }
   
    model_free(model0);
    model_free(model1);
    vertex_buffer_delete(buffer0);

    vertex_buffer_quit();
    glsl_quit();
    sdlkit_quit();
    printf("\ndone.\n");
    return 0;
}
