/*
    vertex_buffer.c
    
    gestion "souple" des vbo

    TODO : 
        . separer l'etat et les bindings du vertex_buffer des manipulations openGL
        . creer un vertex_buffer_state ?
            stockage des formats de donnees + associations attributs de l'objet et attributs de rendu

    mailto:jean-claude.iehl@liris.cnrs.fr
    fevrier 2008
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <GL/glew.h>
#include <GL/gl.h>

#ifndef GL_EXT_gpu_shader4
    #warning no support for glsl_program4
    #define NO_PROGRAM4
#endif

#include "vertex_buffer.h"

/* private api 
 */

#define ATTR_DISABLE -1
#define ATTR_ENABLE 1

#ifndef NDEBUG 
#define checkGLerror() \
{ \
    GLenum code= glGetError(); \
    if(code != GL_NO_ERROR) \
        printf("[%s line %d] GL Error: %s\n",  __FILE__, __LINE__, gluErrorString(code));  \
}

#define printVBOerror(message) \
{ \
    printf("[%s line %d] VBO Error: %s\n",  __FILE__, __LINE__, message);  \
}

#else

#define checkGLerror() 
#define printVBOerror(message)

#endif


/* bind factory */
static 
int disable_pointer(VERTEX_BUFFER *buffer, int flag, int attr_id)
{
    return 0;
}

static 
int vertex_pointer(VERTEX_BUFFER *buffer, int flag, int attr_id)
{
    if(flag == ATTR_DISABLE)
    {
        glDisableClientState(GL_VERTEX_ARRAY);
        return 0;
    }
    
    glEnableClientState(GL_VERTEX_ARRAY);
    glBindBuffer(buffer->attr_mode[attr_id], buffer->names[attr_id]);
    glVertexPointer(buffer->attr_size[attr_id], buffer->attr_type[attr_id], 0, NULL);
    checkGLerror();
    return 0;
}

static 
int index_pointer(VERTEX_BUFFER *buffer, int flag, int attr_id)
{
    if(flag == ATTR_DISABLE)
    {
        glDisableClientState(GL_VERTEX_ARRAY); // ??
        return 0;
    }
    
    glEnableClientState(GL_VERTEX_ARRAY);
    glBindBuffer(buffer->attr_mode[attr_id], buffer->names[attr_id]);
    checkGLerror();
    return 0;
}

static 
int color_pointer(VERTEX_BUFFER *buffer, int flag, int attr_id)
{
    if(flag == ATTR_DISABLE)
    {
        glDisableClientState(GL_COLOR_ARRAY);
        return 0;
    }

    glEnableClientState(GL_COLOR_ARRAY);
    glBindBuffer(buffer->attr_mode[attr_id], buffer->names[attr_id]);
    glColorPointer(buffer->attr_size[attr_id], buffer->attr_type[attr_id], 0, NULL);
    checkGLerror();
    return 0;
}

static 
int normal_pointer(VERTEX_BUFFER *buffer, int flag, int attr_id)
{
    if(flag == ATTR_DISABLE)
    {
        glDisableClientState(GL_NORMAL_ARRAY);
        return 0;
    }

    glEnableClientState(GL_NORMAL_ARRAY);
    glBindBuffer(buffer->attr_mode[attr_id], buffer->names[attr_id]);
    glNormalPointer(buffer->attr_type[attr_id], 0, NULL);
    checkGLerror();
    return 0;
}

static 
int texcoord0_pointer(VERTEX_BUFFER *buffer, int flag, int attr_id)
{
    if(flag == ATTR_DISABLE)
    {
        glClientActiveTexture(GL_TEXTURE0);
        glDisableClientState(GL_TEXTURE_COORD_ARRAY);
        return 0;
    }

    glClientActiveTexture(GL_TEXTURE0);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);
    glBindBuffer(buffer->attr_mode[attr_id], buffer->names[attr_id]);
    glTexCoordPointer(buffer->attr_size[attr_id], buffer->attr_type[attr_id], 0, NULL);
    checkGLerror();
    return 0;
}

static 
int texcoord1_pointer(VERTEX_BUFFER *buffer, int flag, int attr_id)
{
    if(flag == ATTR_DISABLE)
    {
        glClientActiveTexture(GL_TEXTURE1);
        glDisableClientState(GL_TEXTURE_COORD_ARRAY);
        return 0;
    }

    glClientActiveTexture(GL_TEXTURE1);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);
    glBindBuffer(buffer->attr_mode[attr_id], buffer->names[attr_id]);
    glTexCoordPointer(buffer->attr_size[attr_id], buffer->attr_type[attr_id], 0, NULL);
    checkGLerror();
    return 0;
}

static 
int texcoord2_pointer(VERTEX_BUFFER *buffer, int flag, int attr_id)
{
    if(flag == ATTR_DISABLE)
    {
        glClientActiveTexture(GL_TEXTURE2);
        glDisableClientState(GL_TEXTURE_COORD_ARRAY);
        return 0;
    }

    glClientActiveTexture(GL_TEXTURE2);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);
    glBindBuffer(buffer->attr_mode[attr_id], buffer->names[attr_id]);
    glTexCoordPointer(buffer->attr_size[attr_id], buffer->attr_type[attr_id], 0, NULL);
    checkGLerror();
    return 0;
}

static 
int texcoord3_pointer(VERTEX_BUFFER *buffer, int flag, int attr_id)
{
    if(flag == ATTR_DISABLE)
    {
        glClientActiveTexture(GL_TEXTURE3);
        glDisableClientState(GL_TEXTURE_COORD_ARRAY);
        return 0;
    }

    glClientActiveTexture(GL_TEXTURE3);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);
    glBindBuffer(buffer->attr_mode[attr_id], buffer->names[attr_id]);
    glTexCoordPointer(buffer->attr_size[attr_id], buffer->attr_type[attr_id], 0, NULL);
    checkGLerror();
    return 0;
}

static
int vertexattr_pointer(VERTEX_BUFFER *buffer, int flag, int attr_id)
{
    if(flag == ATTR_DISABLE)
    {
        glDisableVertexAttribArray(buffer->binding_locations[attr_id]);
        checkGLerror();
        return 0;
    }

    glEnableVertexAttribArray(buffer->binding_locations[attr_id]);
    glBindBuffer(buffer->attr_mode[attr_id], buffer->names[attr_id]);
    glVertexAttribPointer(buffer->binding_locations[attr_id], 
        buffer->attr_size[attr_id], buffer->attr_type[attr_id], GL_FALSE, 0, NULL);
    checkGLerror();
    return 0;
}

static
int vertexattr_ipointer(VERTEX_BUFFER *buffer, int flag, int attr_id)
{
#ifndef NO_PROGRAM4
    if(flag == ATTR_DISABLE)
    {
        glDisableVertexAttribArray(buffer->binding_locations[attr_id]);
        return 0;
    }

    glEnableVertexAttribArray(buffer->binding_locations[attr_id]);
    checkGLerror();
    glBindBuffer(buffer->attr_mode[attr_id], buffer->names[attr_id]);
    checkGLerror();
    glVertexAttribIPointerEXT(buffer->binding_locations[attr_id], 
        buffer->attr_size[attr_id], buffer->attr_type[attr_id], 0, NULL);
#endif
    checkGLerror();
    return 0;
}
struct _bindf
{
    int name;
    int (*bind)(VERTEX_BUFFER *buffer, int flag, int attr_id);
    char *string;
};

static struct _bindf binding_factory[]=
{
    {DISABLE_ATTR, disable_pointer, "disable"},
    {INDEX_ATTR, index_pointer, "index"},
    {VERTEX_ATTR, vertex_pointer, "vertex"},
    {COLOR_ATTR, color_pointer, "color"},
    {TANGENT_ATTR, disable_pointer, "tangent"},
    {BINORMAL_ATTR, disable_pointer, "binormal"},
    {NORMAL_ATTR, normal_pointer, "normal"},
    {TEXCOORD0_ATTR, texcoord0_pointer, "texcoord0"},
    {TEXCOORD1_ATTR, texcoord1_pointer, "texcoord1"},
    {TEXCOORD2_ATTR, texcoord2_pointer, "texcoord2"},
    {TEXCOORD3_ATTR, texcoord3_pointer, "texcoord3"},
    {VERTEXATTR0_ATTR, vertexattr_pointer, "vertex_attribute0"},
    {VERTEXATTR1_ATTR, vertexattr_pointer, "vertex_attribute1"},
    {VERTEXATTR2_ATTR, vertexattr_pointer, "vertex_attribute2"},
    {VERTEXATTR3_ATTR, vertexattr_pointer, "vertex_attribute3"},
    {OBJECTID_ATTR, vertexattr_ipointer, "object_id"}
};

static int binding_factory_n= sizeof(binding_factory) / sizeof(struct _bindf);

static
char *get_attr_string(GLenum name)
{
    int i;
    
    for(i= 0; i < binding_factory_n; i++)
        if(binding_factory[i].name == name)
            return binding_factory[i].string;
    
    return "(attribute not found)";
}

static
int bind(VERTEX_BUFFER *buffer, int attr_id)
{
    int i;

    if(buffer->bindings[attr_id] == ATTR_DISABLE)
        return 0;

#ifndef NDEBUG
    for(i= 0; i < binding_factory_n; i++)
    {
        if(binding_factory[i].name == attr_id)
        {
            assert(i == attr_id);
            break;
        }
    }
    if(i >= binding_factory_n)
        return -1;
#endif
    
    return binding_factory[attr_id].bind(buffer, ATTR_ENABLE, buffer->bindings[attr_id]);
}

static
int unbind(VERTEX_BUFFER *buffer, int attr_id)
{
    int i;
    
    if(buffer->bindings[attr_id] == ATTR_DISABLE)
        return 0;

#ifndef NDEBUG
    for(i= 0; i < binding_factory_n; i++)
    {
        if(binding_factory[i].name == attr_id)
        {
            assert(i == attr_id);
            break;
        }
    }
    if(i >= binding_factory_n)
        return -1;
#endif
    
    return binding_factory[attr_id].bind(buffer, ATTR_DISABLE, buffer->bindings[attr_id]);
}

static 
int attr_find_name(VERTEX_BUFFER *buffer, int name)
{
    int i;
    
    assert(buffer != NULL);
    
    for(i= 0; i < buffer->n; i++)
        if(buffer->attr_name[i] == name)
            return i;
    
    return -1;
}

static
int attr_find_binding(VERTEX_BUFFER *buffer, int binding)
{
    assert(buffer != NULL);
    
    return buffer->bindings[binding];
}


static void extFailed(char *message)
{
    printf("%-48s  [FAILED]\n", message);
    fflush(stdout);
}

static void extSupported(char *message)
{
#if VERBOSE
    printf("%-48s  [  OK  ]\n", message);
    fflush(stdout);
#endif
}


/* public api 
 */

int vertex_buffer_init(void)
{
    GLenum err;

    err= glewInit();
    if(err != GLEW_OK)
    {
        printf("%s\n", glewGetErrorString(err));
        extFailed("GLEW");
        return -1;
    }
    
    if(!glewIsSupported("GL_VERSION_2_0"))
    {
        extFailed("OpenGL 2");
        return -1;
    }

    if(!glewIsSupported("GL_ARB_vertex_buffer_object"))
    {
        extFailed("vertexbuffer objects");
        return -1;
    }
    
#if VERBOSE_DEBUG
    extSupported("glew");
    extSupported("openGL 2");
#endif
#if VERBOSE
    extSupported("vertexbuffer objects");
#endif
 
    return 0;
}

int vertex_buffer_quit(void)
{
    return 0;
}


VERTEX_BUFFER *vertex_buffer_create_flags(GLenum mode, GLenum usage, int allocate)
{
    VERTEX_BUFFER *buffer= (VERTEX_BUFFER *) malloc(sizeof(VERTEX_BUFFER));
    assert(buffer != NULL);
    
    int i;
    for(i= 0; i < VBO_ATTR_MAX; i++)
        buffer->names[i]= 0;

    for(i= 0; i < ATTR_N; i++)
    {
        buffer->bindings[i]= ATTR_DISABLE;
        buffer->binding_locations[i]= 0;
    }
    
    buffer->mode= mode;
    buffer->usage= usage;
    buffer->allocate= allocate;
    buffer->n= 0;
    
    return buffer;
}

VERTEX_BUFFER *vertex_buffer_create(GLenum mode)
{
    return vertex_buffer_create_flags(mode, GL_STATIC_DRAW, 1);
}


/* detruit le vbo */
int vertex_buffer_delete(VERTEX_BUFFER *buffer)
{
    int i;
    
#ifndef NDEBUG
    if(buffer == NULL)
        return 0;
#endif
    
    if(buffer->allocate != 0)
        for(i= 0; i < buffer->n; i++)
            glDeleteBuffers(1, &buffer->names[i]);
    
    free(buffer);
    
    return 0;
}

/* verifie le binding du vbo */
int vertex_buffer_check(VERTEX_BUFFER *buffer)
{
    // verifie qu'il y a au moins de la geometrie a dessiner
    if(attr_find_binding(buffer, VERTEX_ATTR) < 0)
        return -1;
    
    // FIX : a completer
    
    return 0;
}

/* affichage des attributs selectionnes */
int vertex_buffer_draw_attributes(VERTEX_BUFFER *buffer, unsigned int flags)
{
    int vertex_binding;
    int index_binding;
    int i, id;
    
#ifndef NDEBUG
    if(buffer == NULL)
        return -1;
    if(buffer->n == 0)
        return -1;
#endif
    // ne pas essayer de dessiner un buffer non alloue, c'est un autre gestionnaire qui le fera
    if(buffer->allocate == 0)
        return -1;
    
    /* bind les attributs selectionnes */
    for(i= 0; i < ATTR_N; i++)
    {
        if(ATTR_BIT(i) & flags) 
        {
        #ifndef NDEBUG
            id= buffer->bindings[i];
            if(id == ATTR_DISABLE)
                continue;
            
            if(buffer->names[id] == 0)
            {
                printVBOerror("no buffer allocated");
                return -1;
            }
        #endif
            
            if(bind(buffer, i) < 0)
                return -1;
        }
    }
    
    // recuperer vertex_attr et index_attr
    vertex_binding= attr_find_binding(buffer, VERTEX_ATTR);
    if(vertex_binding < 0)
        return -1;
    
    index_binding= attr_find_binding(buffer, INDEX_ATTR);
    if(index_binding < 0)
        glDrawArrays(buffer->mode, 
            0, buffer->attr_data_n[vertex_binding]);
    else
        glDrawElements(buffer->mode, 
            buffer->attr_data_n[index_binding], buffer->attr_type[index_binding], NULL);
    
    checkGLerror();    
    return 0;
}

/* affichage de tous les attributs */
int vertex_buffer_draw(VERTEX_BUFFER *buffer)
{
    return vertex_buffer_draw_attributes(buffer, ALL_ATTR_BIT);
}

/* annule tous les bindings du vbo */
int vertex_buffer_disable(VERTEX_BUFFER *buffer)
{
    int i;
    
#ifndef NDEBUG
    if(buffer == NULL)
        return -1;
    if(buffer->n == 0)
        return -1;
#endif
    
    if(buffer->allocate == 0)
        return -1;
    
    for(i= 0; i < ATTR_N; i++)
        unbind(buffer, i);
    
    return 0;
}


//~ static 
int get_vbo_elem_size(int size, int type)
{
    int type_size= 0;
    
    switch(type)
    {
        case GL_UNSIGNED_BYTE:
        case GL_BYTE:
            type_size= sizeof(char);
        break;
        case GL_UNSIGNED_SHORT:
        case GL_SHORT:
            type_size= sizeof(short int);
        break;
        case GL_UNSIGNED_INT:
        case GL_INT:
            type_size= sizeof(int);
        break;
        case GL_FLOAT:
            type_size= sizeof(float);
        break;
        case GL_DOUBLE:
            type_size= sizeof(double);
        break;
        default:
            type_size= 0;
        break;
    }
    
    return type_size * size;
}

static 
int attr_add(VERTEX_BUFFER *buffer, GLenum mode, int name, 
    GLint elem_size, GLenum elem_type, int elem_n, GLsizei stride, void *vdata)
{
    unsigned char *map;
    unsigned char *data;
    unsigned int vbo_elem_size;
    unsigned int vbo_size;
    int id= buffer->n++;
    
    buffer->attr_mode[id]= mode;
    buffer->attr_name[id]= name;
    buffer->attr_size[id]= elem_size;
    buffer->attr_type[id]= elem_type;
    buffer->attr_stride[id]= stride;
    buffer->attr_data_n[id]= elem_n;
    buffer->attr_data[id]= vdata;
    
    // binding par defaut
    buffer->bindings[name]= id;
    buffer->binding_locations[name]= 0;

    // allouer le vbo
    if(buffer->allocate == 0)
        return 0;

    glGenBuffers(1, &buffer->names[id]);
    glBindBuffer(mode, buffer->names[id]);
    checkGLerror();

    // calculer la taille du vbo
    vbo_elem_size= get_vbo_elem_size(elem_size, elem_type);
    vbo_size= vbo_elem_size * elem_n;
    glBufferData(mode, vbo_size, NULL, buffer->usage);
    checkGLerror();

    data= (unsigned char *) vdata;
    map= (unsigned char *) glMapBuffer(mode, GL_WRITE_ONLY);
    checkGLerror();
    if(map == NULL)
        return -1;
    
    // copier les donnees dans le buffer
    if(stride == 0 || stride == vbo_elem_size)
        // les donnees sont contigues ...
        memcpy(map, data, vbo_size);
    
    else
    {
        // copie element par element + stride
        unsigned int map_offset;
        unsigned int data_offset;
        int i;
        
        map_offset= 0;
        data_offset= 0;
        for(i= 0; i < elem_n; i++)
        {
            memcpy(&map[map_offset], &data[data_offset], vbo_elem_size);
            map_offset+= vbo_elem_size;
            data_offset+= stride;
        }
    }
    
    glUnmapBuffer(mode);
    
    return 0;
}

/* attache un attribut au vbo */
int vertex_buffer_add_attribute(VERTEX_BUFFER *buffer, int name, 
    GLint elem_size, GLenum elem_type, int elem_n, GLsizei stride, void *data)
{
#ifndef NDEBUG
    if(buffer == NULL)
        return -1;
    if(buffer->n >= VBO_ATTR_MAX)
        return -1;
    if(name < 0 || name >= ATTR_N)
        return -1;
    
    // verifie que l'attribut n'a pas deja ete attache
    if(attr_find_name(buffer, name) >= 0)
        return -1;
#endif

    return attr_add(buffer, GL_ARRAY_BUFFER, name, elem_size, elem_type, elem_n, stride, data);
}

/* attache un index buffer */
int vertex_buffer_add_index(VERTEX_BUFFER *buffer, 
    GLenum elem_type, int elem_n, GLsizei stride, void *data)
{
#ifndef NDEBUG
    if(buffer == NULL)
        return -1;
    if(buffer->n >= VBO_ATTR_MAX)
        return -1;
#endif
    
    return attr_add(buffer, GL_ELEMENT_ARRAY_BUFFER, INDEX_ATTR, 1, elem_type, elem_n, stride, data);
}

/* precise comment utiliser un attribut, version souple
    un attribut de l'objet peut etre associe a plusieurs attributs de rendu
 */
int vertex_buffer_rebind_attribute(VERTEX_BUFFER *buffer, int name, int attr)
{
    int id;
    
#ifndef NDEBUG
    if(buffer == NULL)
        return -1;
    if(name < 0 || name >= ATTR_N)
        return -1;
    if(attr < 0 || attr >= ATTR_N)
        return -1;
#endif
    
    // verifier que l'attribut existe 
    id= attr_find_name(buffer, name);
    if(id < 0)
    {
        printVBOerror("attribute not found");
        return -1;
    }
    
    // stocke le binding
    if(attr != DISABLE_ATTR)
    {
        buffer->bindings[attr]= id;
        buffer->binding_locations[attr]= 0;
    }
    else
    {
        buffer->bindings[name]= ATTR_DISABLE;
        buffer->binding_locations[name]= 0;
    }
    
    return 0;
}

/* precise comment utiliser un attribut, version simple 
    un attribut de l'objet ne peut etre associe qu'a un seul attribut de rendu
 */
int vertex_buffer_bind_attribute(VERTEX_BUFFER *buffer, int name, int attr)
{
    // desactive l'attribut 'source'
    if(vertex_buffer_rebind_attribute(buffer, name, DISABLE_ATTR) < 0)
        return -1;
    
    // associe l'attribut de l'objet a un attribut de rendu
    return vertex_buffer_rebind_attribute(buffer, name, attr);
    // ... une seule association possible
}


/* precise comment utiliser un attribut */
int vertex_buffer_bind_attribute_location(VERTEX_BUFFER *buffer, int name, GLuint location)
{
    int id;
    
#ifndef NDEBUG
    if(buffer == NULL)
        return -1;
    if(name < 0 || name >= ATTR_N)
        return -1;
#endif
    
    // verifier que l'attribut existe 
    id= attr_find_name(buffer, name);
    if(id < 0)
    {
        printVBOerror("attribute not found");
        return -1;
    }
    
    // stocke le binding
    buffer->binding_locations[id]= location;
    return 0;
}

/* retablit le mapping par defaut : VERTEX_ATTR -> glVertexPointer, etc. */
int vertex_buffer_bind_disable(VERTEX_BUFFER *buffer)
{
    int i;
    
#ifndef NDEBUG
    if(buffer == NULL)
        return -1;
#endif
    
    // desactive le binding precedent
    if(buffer->allocate != 0)
        for(i= 0; i < ATTR_N; i++)
            unbind(buffer, i);

    // retablit les bindings par defaut
    for(i= 0; i < ATTR_N; i++)
    {
        buffer->bindings[i]= attr_find_name(buffer, i);
        if(buffer->bindings[i] < 0)
            buffer->bindings[i]= ATTR_DISABLE;
        buffer->binding_locations[i]= 0;
    }
    
    return 0;
}

