/*
    gestion des vertex_buffers
    construction de batchs
    
    mailto:jean-claude.iehl@liris.cnrs.fr
    fevrier 2008

    // FIX : allouer un batch d'une taille fixe et afficher le batch chaque fois qu'il est plein !!
    // FIX : eliminer les copies inutiles
    // TODO : verification du type de primitive et des bindings == tout doit etre coherent

 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <GL/glew.h>
#include <GL/gl.h>

#include "mat44.h"
#include "vertex_buffer.h"
#include "vertex_batch.h"

/* private api 
 */

#define ATTR_DISABLE -1
#define ATTR_ENABLE 1

#ifndef NDEBUG 
#define checkGLerror() \
{ \
    GLenum code= glGetError(); \
    if(code != GL_NO_ERROR) \
        printf("[%s line %d] GL Error: %s\n",  __FILE__, __LINE__, gluErrorString(code));  \
}

#define printVBOerror(message) \
{ \
    printf("[%s line %d] VBO Error: %s\n",  __FILE__, __LINE__, message);  \
}

#else

#define checkGLerror() 
#define printVBOerror(message)

#endif

/* public api
 */
 

/*
    vertex_batch_create()
    vertex_batch_bind_object_id_location(, glGetAttribLocation(program, "object_id"))
    vertex_batch_bind_model_matrix_location(, glGetUniformLocation(program, "model_matrix"))

    vertex_batch_begin()
    for(objects to draw)
    {
        if(vertex_batch_add() < 0)
            vertex_batch_draw() | vertex_batch_draw_attributes()
    }
    vertex_batch_end()
    vertex_batch_draw() | vertex_batch_draw_attributes()
    
 */

VERTEX_BATCH *vertex_batch_create(GLenum mode, int size)
{
    VERTEX_BATCH *batch= (VERTEX_BATCH *) malloc(sizeof(VERTEX_BATCH));
    assert(batch != NULL);
    batch->size= size;
    batch->n= 0;
    batch->mode= mode;
    batch->object_id_location= 0;
    batch->model_matrix_location= 0;
    
    batch->batch= NULL;
    
    batch->buffers= (VERTEX_BUFFER **) malloc(sizeof(VERTEX_BUFFER *) * size);
    assert(batch->buffers != NULL);
    batch->model_matrix= (MAT44 *) malloc(sizeof(MAT44) * size);
    assert(batch->model_matrix != NULL);
    
    return batch;
}

int vertex_batch_delete(VERTEX_BATCH *batch)
{
    if(batch == NULL)
        return 0;
    
    if(batch->batch != NULL)
        vertex_buffer_delete(batch->batch);
    if(batch->buffers != NULL)
        free(batch->buffers);
    if(batch->model_matrix != NULL)
        free(batch->model_matrix);

    free(batch);
    return 0;
}

int vertex_batch_bind_object_id_location(VERTEX_BATCH *batch, GLint location)
{
    if(batch == NULL)
        return -1;
  
    batch->object_id_location= location;
    return 0;
}


int vertex_batch_bind_model_matrix_location(VERTEX_BATCH *batch, GLint location)
{
    if(batch == NULL)
        return -1;
    
    batch->model_matrix_location= location;
    return 0;
}

int vertex_batch_begin(VERTEX_BATCH *batch)
{
    if(batch == NULL)
        return -1;
    
    if(batch->batch != NULL)
    {
        // FIX : allouer un batch d'une taille fixe et afficher le batch chaque fois qu'il est plein !!
        vertex_buffer_delete(batch->batch);
        batch->batch= NULL;
    }
    batch->n= 0;
    
    return 0;
}

static
void fill_buffer(unsigned char *map, int elem_size, int elem_n, int stride, unsigned char *data)
{
    int size= elem_size * elem_n;
    
    // copier les donnees dans le buffer
    if(stride == 0 || stride == elem_size)
        // les donnees sont contigues ...
        memcpy(map, data, size);
    
    else
    {
        // copie element par element + stride
        unsigned int map_offset;
        unsigned int data_offset;
        int i;
        
        map_offset= 0;
        data_offset= 0;
        for(i= 0; i < elem_n; i++)
        {
            memcpy(&map[map_offset], &data[data_offset], elem_size);
            map_offset+= elem_size;
            data_offset+= stride;
        }
    }
}

static
int batch_bind(VERTEX_BATCH *batch)
{
    // private vertex_buffer api 
    extern int get_vbo_elem_size(int size, int type);
    
    int vertex_size= 0;
    int vertex_n= 0;
    int index_size= 0;
    int index_n= 0;
    int elem_size;
    int size;
    
    int vertex_id, index_id;
    int i, k;
    
    assert(batch != NULL);
    if(batch->n == 0)
        return 0;
    
    // determiner la taille du batch : nombre de sommets + nombre d'indices
    for(i= 0; i < batch->n; i++)
    {
        // recuperer l'attribut associe a POSITION
        vertex_id= batch->buffers[i]->bindings[VERTEX_ATTR];
        assert(vertex_id >= 0);

        elem_size= get_vbo_elem_size(batch->buffers[i]->attr_size[vertex_id], 
            batch->buffers[i]->attr_type[vertex_id]);
        size= elem_size * batch->buffers[i]->attr_data_n[vertex_id];
        vertex_size+= size;
        vertex_n+= batch->buffers[i]->attr_data_n[vertex_id];
        // fix
        assert(batch->buffers[i]->attr_size[vertex_id] == 3);
        assert(batch->buffers[i]->attr_type[vertex_id] == GL_FLOAT);
        
        // recuperer l'attribut associe a INDEX
        index_id=  batch->buffers[i]->bindings[INDEX_ATTR];
        assert(index_id >= 0);
        elem_size= get_vbo_elem_size(batch->buffers[i]->attr_size[index_id], 
            batch->buffers[i]->attr_type[index_id]);
        size= elem_size * batch->buffers[i]->attr_data_n[index_id];
        index_size+= size;
        // fix
        assert(batch->buffers[i]->attr_size[index_id] == 1);
        assert(batch->buffers[i]->attr_type[index_id] == GL_INT
        || batch->buffers[i]->attr_type[index_id] == GL_UNSIGNED_INT);
    }
    
    // alloue le batch
    // FIX : eliminer les copies inutiles et les malloc / free
    unsigned char *vertex= (unsigned char *) malloc(vertex_size);
    assert(vertex != NULL);
    unsigned char *index= (unsigned char *) malloc(index_size);
    assert(index != NULL);
    unsigned char *object_id= (unsigned char *) malloc(sizeof(int) * vertex_n);
    assert(object_id != NULL);
    
    vertex_size= 0;
    vertex_n= 0;
    index_size= 0;
    index_n= 0;
    for(i= 0; i < batch->n; i++)
    {
        // copie les positions 
        vertex_id= batch->buffers[i]->bindings[VERTEX_ATTR];
        elem_size= get_vbo_elem_size(batch->buffers[i]->attr_size[vertex_id], 
            batch->buffers[i]->attr_type[vertex_id]);
        fill_buffer(&vertex[vertex_size], 
            elem_size, 
            batch->buffers[i]->attr_data_n[vertex_id], 
            batch->buffers[i]->attr_stride[vertex_id], 
            (unsigned char *) batch->buffers[i]->attr_data[vertex_id]);
    
        // copie les indices
        index_id=  batch->buffers[i]->bindings[INDEX_ATTR];
        elem_size= get_vbo_elem_size(batch->buffers[i]->attr_size[index_id], 
            batch->buffers[i]->attr_type[index_id]);
        fill_buffer(&index[index_size], 
            elem_size, 
            batch->buffers[i]->attr_data_n[index_id], 
            batch->buffers[i]->attr_stride[index_id], 
            (unsigned char *) batch->buffers[i]->attr_data[index_id]);
        
        // renumerote les indices 
        //~ printf("batch: remap index %d %d\n", 0, vertex_n);
        int *pindex= (int *) &index[index_size];
        for(k= 0; k < batch->buffers[i]->attr_data_n[index_id]; k++)
            pindex[k]+= vertex_n;
        
        // affecte object_id
        int *pobject_id= (int *) &object_id[vertex_n * sizeof(int)];
        for(k= 0; k < batch->buffers[i]->attr_data_n[vertex_id]; k++)
            pobject_id[k]= i;

        elem_size= get_vbo_elem_size(batch->buffers[i]->attr_size[vertex_id], 
            batch->buffers[i]->attr_type[vertex_id]);
        size= elem_size * batch->buffers[i]->attr_data_n[vertex_id];
        vertex_size+= size;
        vertex_n+= batch->buffers[i]->attr_data_n[vertex_id];

        elem_size= get_vbo_elem_size(batch->buffers[i]->attr_size[index_id], 
            batch->buffers[i]->attr_type[index_id]);
        size= elem_size * batch->buffers[i]->attr_data_n[index_id];
        index_size+= size;
        index_n+= batch->buffers[i]->attr_data_n[index_id];
    }
    
    // charge les vbo + index + object_id
    if(batch->batch != NULL)
        // FIX : allouer un batch d'une taille fixe et afficher le batch chaque fois qu'il est plein !!
        vertex_buffer_delete(batch->batch);
    batch->batch= vertex_buffer_create_flags(batch->mode, GL_STATIC_DRAW, 1);
    assert(batch->batch != NULL);
    
    //~ printf("batch: vertex_n %d, index_n %d\n", vertex_n, index_n);
    
    int code;
    code= vertex_buffer_add_attribute(batch->batch, VERTEX_ATTR, 3, GL_FLOAT, 
        vertex_n, 0, vertex);
    assert(code == 0);
    code= vertex_buffer_add_index(batch->batch, GL_UNSIGNED_INT, 
        index_n, 0, index);
    assert(code == 0);
    code= vertex_buffer_add_attribute(batch->batch, OBJECTID_ATTR, 1, GL_INT, 
        vertex_n, 0, object_id);
    assert(code == 0);
    
    // bind object_id
    vertex_buffer_bind_attribute_location(batch->batch, OBJECTID_ATTR, batch->object_id_location);
    
    free(object_id);
    free(index);
    free(vertex);
    return 0;
}

int vertex_batch_end(VERTEX_BATCH *batch)
{
    if(batch == NULL)
        return -1;
    
    batch_bind(batch);
    batch->n= 0;
    
    return 0;
}

int vertex_batch_add(VERTEX_BATCH *batch, VERTEX_BUFFER *buffer, MAT44 model_matrix)
{
    if(batch == NULL)
        return -1;
    
    if(batch->n >= batch->size)
        return -1;      // indiquer que le batch est plein
    
    // TODO : verification du type de primitive et des bindings == tout doit etre coherent

    mat44_copy(&batch->model_matrix[batch->n][0], model_matrix);
    batch->buffers[batch->n]= buffer;
    batch->n++;

    return 0;
}

int vertex_batch_draw_attributes(VERTEX_BATCH *batch, unsigned int flags)
{
    int code;
    if(batch == NULL)
        return -1;

    if(batch->n > 0)
        batch_bind(batch);
    checkGLerror();

    {
        // transmettre les matrices (seules les n premieres sont intialisees ...)
        glUniformMatrix4fv(batch->model_matrix_location, (GLsizei) batch->size, GL_FALSE, &batch->model_matrix[0][0]);
        // afficher le batch
        code= vertex_buffer_draw_attributes(batch->batch, flags);
    }
    //~ {
        //~ glMatrixMode(GL_MODELVIEW);
        //~ glPushMatrix();
        //~ glMultMatrixf(&batch->model_matrix[0][0]);
        //~ code= vertex_buffer_draw_attributes(batch->batch, flags);
        //~ glPopMatrix();
    //~ }
    checkGLerror();

    // FIX
    vertex_buffer_delete(batch->batch);
    batch->batch= NULL;
    batch->n= 0;
    
    return code;
}

int vertex_batch_draw(VERTEX_BATCH *batch)
{
    return vertex_batch_draw_attributes(batch, ALL_ATTR_BIT);
}

