/*
    utilitaires GLSL
    
    mailto:jciehl@bat710.univ-lyon1.fr
    
    janvier 2007
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include <GL/glew.h>
#include <GL/gl.h>

#include "glsl2.h"

void extFailed(char *message)
{
    printf("%-48s  [FAILED]\n", message);
    fflush(stdout);
}

void extSupported(char *message)
{
#if VERBOSE
    printf("%-48s  [  OK  ]\n", message);
    fflush(stdout);
#endif
}


int extIsSupported(char *ext, char *message, int verbose)
{
    if(glewIsSupported(ext))
    {
        if(verbose > 0)
            extSupported(message);
        return 0;
    }
    else
    {
#if VERBOSE_DEBUG	// n'affiche pas les tentatives
        if(verbose > 0)
            extFailed(message);
#endif
        return -1;	
    }
}


int glsl_load_source(char **str, int *n, char *fname)
{
    FILE *in;
    int size;
    
    *n= 0;
    *str= NULL;
    size= 0;		

    if(fname==NULL)
        return 0;
    in= fopen(fname, "r");
    if(in==NULL)
    {
    #if VERBOSE
        char tmp[1024];
        snprintf(tmp, sizeof(tmp), "loading shader '%s'...", fname);
        tmp[1023]= 0;
        extFailed(tmp);
    #endif
        
        return -1;
    }
    
    #if VERBOSE
        char tmp[1024];
        snprintf(tmp, sizeof(tmp), "loading shader '%s'...", fname);
        tmp[1023]= 0;
        extSupported(tmp);
    #endif
    
    do
    {		
        size+= 4096;
        *str= (char *) realloc(*str, size);
        assert(*str != NULL);

        (*n)+= (int) fread(*str + *n, 1, 4096, in);
    }
    while(!feof(in));
    
    fclose(in);
    (*str)[*n]= 0;
    return 0;
}


void glsl_program_info(GLuint program)
{
    char *log;
    int length= 0;

    glGetProgramiv(program, GL_INFO_LOG_LENGTH, &length);
    if(length > 0)
    {
        log= malloc(length+1);
        assert(log != NULL);
        
        glGetProgramInfoLog(program, length, &length, log);
        log[length]= 0;
        printf("\nerrors:\n%s\n", log);
        free(log);
    }
}

void glsl_shader_info(GLuint shader)
{
    char *log;
    int length= 0;

    glGetShaderiv(shader, GL_INFO_LOG_LENGTH, &length);
    if(length > 0)
    {
        log= malloc(length+1);
        assert(log != NULL);
        
        glGetShaderInfoLog(shader, length, &length, log);
        log[length]= 0;
        printf("\nerrors:\n%s\n", log);
        free(log);
    }
}

static void display_shader_source(const GLchar *source, int line_nb)
{
    int i= 0;
    int n= 1;

    while(source[i] != 0 && n < line_nb)
    {
        if(source[i] == '\r' || (source[i] == '\n' && source[i -1] != '\r'))
            n++;
        i++;
    }

    printf("\t");
    while(source[i] != 0)
    {
        if(source[i] == '\r' || (source[i]=='\n' && source[i -1] != '\r'))
            break;
        printf("%c", source[i]);
        i++;
    }
    
    printf("\n");
}


void glsl_shader_source_info(GLuint shader, char *fname, const GLchar *source, GLenum type)
{
    char *log;
    char *line;
    int shader_nb, line_nb, line_break, prev_line_nb;
    int length= 0;

    printf("shader source '%s':\n\n", fname);
    
    glGetShaderiv(shader, GL_INFO_LOG_LENGTH, &length);
    if(length > 0)
    {
        log= malloc(length+1);
        assert(log != NULL);
        
        glGetShaderInfoLog(shader, length, &length, log);
        log[length]= 0;
        //~ printf("\nerrors:\n%s\n", log);
        
        line= log;
        prev_line_nb= 0;
        for(;;)
        {
            line_break= 0;
            if(sscanf(line, " (%d) : %n", &line_nb, &line_break) < 1    // nvidia syntax
            && sscanf(line, " ERROR : %d : %d : %n", &shader_nb, &line_nb, &line_break) < 1)    // ati syntax
                break;

            if(line_nb != prev_line_nb)
            {
                prev_line_nb= line_nb;
                display_shader_source(source, line_nb);
            }
            
            printf("%s:%d: ", fname, line_nb);
            while(line[line_break] != 0
            && line[line_break] != '\r' && line[line_break] != '\n')
            {
                printf("%c", line[line_break]);
                line_break++;
            }
            printf("\n");
            line= &line[line_break +1];
        }
        
        free(log);
    }
}


int glsl_add_shader(GLuint program, char *fname, const GLchar *source, GLenum type, int version)
{
    GLenum err;
    GLuint shader;
    GLint compiled;
    
    assert(program != 0);
    assert(type != 0);
    
    if(source == NULL)
        return 0;
    
    shader= glCreateShader(type);
    assert(shader != 0);
    
#if 1
    const GLchar *sources110[]= 
    {
        "#version 110\n",
        "#line 0\n",
        source
    };
    const GLsizei sources110_n= sizeof(sources110) / sizeof(GLchar *);

    const GLchar *sources120[]= 
    {
        "#version 120\n",
        "#line 0\n",
        source
    };
    const GLsizei sources120_n= sizeof(sources120) / sizeof(GLchar *);

    if(version == 120)
        glShaderSource(shader, sources120_n, sources120, NULL);
    else
        glShaderSource(shader, sources110_n, sources110, NULL);

#else
    glShaderSource(shader, 1, (const GLchar **) &source, NULL);
#endif

    glCompileShader(shader);

    compiled= GL_FALSE;
    glGetShaderiv(shader, GL_COMPILE_STATUS, &compiled);
    if(compiled == GL_FALSE)
    {
        if(type == GL_VERTEX_SHADER)
            extFailed("vertex shader compilation");
        else if(type == GL_FRAGMENT_SHADER)
            extFailed("fragment shader compilation");
    #ifndef NO_PROGRAM4
        else if(type == GL_GEOMETRY_SHADER_EXT)
            extFailed("geometry shader compilation");
    #endif
        else
            extFailed("shader? compilation");
        
        //~ glsl_shader_info(shader);
        glsl_shader_source_info(shader, fname, source, type);
        return -1;
    }
#if VERBOSE
    else
    {
        if(type == GL_VERTEX_SHADER)
            extSupported("vertex shader compilation");
        else if(type == GL_FRAGMENT_SHADER)
            extSupported("fragment shader compilation");
    #ifndef NO_PROGRAM4
        else if(type == GL_GEOMETRY_SHADER_EXT)
            extFailed("geometry shader compilation");
    #endif
    }
#endif
    
    glAttachShader(program, shader);
    glDeleteShader(shader);
    
    err= glGetError();
    if(err != GL_NO_ERROR)
    {
        fprintf(stderr, "\nopenGL Error  %s:%d\n%s\n", __FILE__, __LINE__, gluErrorString(err));
        return -1;
    }

    return 0;
}

GLuint glsl_program_init(char *vfname, char *ffname)
{
    GLchar *source;
    GLuint program;
    GLint linked;
    int length;
    int code;
        
    program= glCreateProgram();
    
    // vertex shader
    if(glsl_load_source(&source, &length, vfname) < 0)
        return 0;
    code= glsl_add_shader(program, vfname, source, GL_VERTEX_SHADER, 120);
    if(source != NULL)
        free(source);
    if(code < 0)
        return 0;
    
    // fragment shader
    if(glsl_load_source(&source, &length, ffname) < 0)
        return 0;
    code= glsl_add_shader(program, ffname, source, GL_FRAGMENT_SHADER, 120);
    if(source != NULL)
        free(source);
    if(code < 0)
        return 0;

    // link
    glLinkProgram(program);
    linked= GL_FALSE;
    glGetProgramiv(program, GL_LINK_STATUS, &linked);
    if(linked == GL_FALSE)
    {
        extFailed("program link");
        glsl_program_info(program);
        return 0;
    }
#if VERBOSE_DEBUG
    else
        extSupported("program link");
#endif
    
    return program;
}


static 
GLuint program_get_shader(GLuint program, GLenum type)
{
    GLint n;
    glGetProgramiv(program, GL_ATTACHED_SHADERS, &n);
    if(n == 0)
        return 0;
    
    GLuint shaders[n];
    glGetAttachedShaders(program, n, &n, shaders);
        
    int i;
    for(i= 0; i < n; i++)
    {
        GLint shader_type;
        glGetShaderiv(shaders[i], GL_SHADER_TYPE, &shader_type);
        if(shader_type == type)
            return shaders[i];
    }
    
    return 0;
}

GLuint glsl_program_get_vertex_shader(GLuint program)
{
    return program_get_shader(program, GL_VERTEX_SHADER);
}

GLuint glsl_program_get_fragment_shader(GLuint program)
{
    return program_get_shader(program, GL_FRAGMENT_SHADER);
}

// validate
int glsl_program_validate(GLuint program)
{
    GLint validated;
    
    if(program == 0)
        return 0;
    
    glValidateProgram(program);
    validated= GL_FALSE;
    glGetProgramiv(program, GL_VALIDATE_STATUS, &validated);
    if(validated == GL_FALSE)
    {
        extFailed("program validation");
        glsl_program_info(program);
        return -1;
    }
#if VERBOSE_DEBUG
    else
        extSupported("program validation");
#endif
    
    return 0;
}

int glsl_init(void)
{
    GLenum err;

    err= glewInit();
    if(err != GLEW_OK)
    {
        printf("%s\n", glewGetErrorString(err));
        extFailed("glew");
        return -1;
    }
    
    if(!glewIsSupported("GL_VERSION_2_0"))
    {
        extFailed("openGL 2");
        return -1;
    }
    
#if VERBOSE_DEBUG
    extSupported("glew");
    extSupported("openGL 2");
#endif
    
    return 0;
}

int glsl_quit(void)
{
    return 0;
}



#ifndef NO_PROGRAM4
GLuint glsl_program4_init(char *vfname, char *gfname, char *ffname)
{
    GLchar *source;
    GLuint program;
    GLint linked;
    int length;
    int code;
        
    program= glCreateProgram();
    
    // vertex shader
    if(glsl_load_source(&source, &length, vfname) < 0)
        return 0;
    code= glsl_add_shader(program, vfname, source, GL_VERTEX_SHADER, 120);
    if(source != NULL)
        free(source);
    if(code < 0)
        return 0;

    // geometry shader
    if(glsl_load_source(&source, &length, gfname) < 0)
        return 0;
    code= glsl_add_shader(program, gfname, source, GL_GEOMETRY_SHADER_EXT, 120);
    if(source != NULL)
        free(source);
    if(code < 0)
        return 0;
    
    // fragment shader
    if(glsl_load_source(&source, &length, ffname) < 0)
        return 0;
    code= glsl_add_shader(program, ffname, source, GL_FRAGMENT_SHADER, 120);
    if(source != NULL)
        free(source);
    if(code < 0)
        return 0;

    // link
    glLinkProgram(program);
    linked= GL_FALSE;
    glGetProgramiv(program, GL_LINK_STATUS, &linked);
    if(linked == GL_FALSE)
    {
        extFailed("program4 link");
        glsl_program_info(program);
        return 0;
    }
#if VERBOSE_DEBUG
    else
        extSupported("program4 link");
#endif
    
    return program;
}

int glsl_program4_link(GLuint program)
{
    GLint linked;

    glLinkProgram(program);
    linked= GL_FALSE;
    glGetProgramiv(program, GL_LINK_STATUS, &linked);
    if(linked == GL_FALSE)
    {
        extFailed("program4 link");
        glsl_program_info(program);
        return -1;
    }
#if VERBOSE_DEBUG
    else
        extSupported("program4 link");
#endif

    return 0;
}


GLuint glsl_program4_get_vertex_shader(GLuint program)
{
    return program_get_shader(program, GL_VERTEX_SHADER);
}

GLuint glsl_program4_get_geometry_shader(GLuint program)
{
    return program_get_shader(program, GL_GEOMETRY_SHADER_EXT);
}

GLuint glsl_program4_get_fragment_shader(GLuint program)
{
    return program_get_shader(program, GL_VERTEX_SHADER);
}

int glsl_vertex_shader_get_max_uniforms(void)
{
    int n;

    glGetIntegerv(GL_MAX_VERTEX_UNIFORM_COMPONENTS, &n);
    return n;
}

int glsl_program4_get_free_uniforms(GLuint program)
{
    return 0;
}


int glsl_init4(void)
{
    GLenum err;

    err= glewInit();
    if(err != GLEW_OK)
    {
        printf("%s\n", glewGetErrorString(err));
        extFailed("glew");
        return -1;
    }
    
    if(!glewIsSupported("GL_VERSION_2_0 GL_EXT_gpu_shader4 GL_EXT_geometry_shader4"))
    {
        extFailed("shader4");
        return -1;
    }
    
#if VERBOSE_DEBUG
    extSupported("glew");
    extSupported("openGL 2");
    extSupported("shader4");
#endif
    
    return 0;
}

#else

GLuint glsl_program4_init(char *vfname, char *gfname, char *ffname)
{
    extFailed("program4");
    return 0;
}

int glsl_init4(void)
{
    extFailed("shader4");
    return -1;
}

#endif
