/*
	scenes maya obj 3, ascii, triangulees
	
	mailto:jciehl@bat710.univ-lyon1.fr
	fevrier 2004 : premiere version
	fevrier 2006 : 
		stockage des attributs compatible avec vertex_buffer_objects
		triangulation du modele
		nettoyage du modele
        janvier 2008 : 
                subdivision recursive du modele (aire des triangles < aire_max)
                interface / accesseurs supplementaires
*/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <math.h>

#include "vec.h"
#include "model.h"


// tableaux dynamiques
void array_add(void **parray, int *size, int n, int e_size, int e_more)
{
    if(n >= *size)
    {
        *size= ((n / e_more) + 1) * e_more;
        
        *parray= realloc(*parray, *size * e_size);
        assert(*parray!=NULL);
    }
}

// gestion des structures
MODEL *model_new(char *filename)
{
    MODEL *m;
    
    m= (MODEL *) malloc(sizeof(MODEL));	
    assert(m!=NULL);

#if 0
    char *s, *bs;
    
    s= strrchr(filename, '/');
    bs= strrchr(filename, '\\');
    if(s==NULL && bs!=NULL)
        s= bs;
    
    if(s!=NULL)
        m->filename= strdup(&s[1]);
    else
        m->filename= strdup(filename);

#else
    // fix: permet d'enregistrer les preferences dans le repertoire du modele
    m->filename= strdup(filename);
#endif

    assert(m->filename);

    m->flags= 0;
    m->status= 0;
    
    m->groups= NULL;
    m->groups_size= 0;
    m->groups_n= 0;
    
    m->faces= NULL;
    m->faces_size= 0;
    m->faces_n= 0;
    
    m->draw_index= NULL;
    m->draw_index_n= 0;
    
    m->mat= NULL;
    m->mat_size= 0;
    m->mat_n= 0;
    
    m->attr= NULL;
    m->attr_size= 0;
    m->attr_n= 0;
    
    m->v= NULL;
    m->v_size= 0;
    m->v_n= 0;
    
    m->norm= NULL;
    m->norm_size= 0;
    m->norm_n= 0;
    
    m->tex= NULL;
    m->tex_size= 0;
    m->tex_n= 0;
    
    m->param= NULL;
    m->param_size= 0;
    m->param_n= 0;
    
    return m;
}

void model_print_stat(MODEL *model)
{
	int triangles_n, quads_n, polys_n;
	int i;
	
	printf("model '%s' : %d (%d) groups, %d (%d) faces, %d (%d) vertices\n", 
		model->filename, 
		model->groups_n, model->groups_size, 
		model->faces_n, model->faces_size, 
		model->v_n, model->v_size);

	triangles_n= 0;
	quads_n= 0;
	polys_n= 0;
	for(i= 0; i < model->faces_n; i++)
	{
		if(model->faces[i].attr < 0)
			continue;
		
		if(model->faces[i].n==3)
			triangles_n++;
		else if(model->faces[i].n==4)
			quads_n++;
		else
			polys_n++;
	}
	
	printf("%d triangles, %d quads, %d polygons\n", triangles_n, quads_n, polys_n);
	
	printf("materials %d (%d) :\n", model->mat_n, model->mat_size);
	for(i= 0; i < model->mat_n; i++)
		printf("  %s\n", model->mat[i].name);
}

void material_free(MATERIAL *m, int n)
{
	int i;
	
	if(m==NULL)
		return;
	
	for(i= 0; i < n; i++)
		if(m[i].name!=NULL)
			free(m[i].name);
	
	free(m);
}

void face_free(FACE *f, int n)
{
	if(f==NULL)
		return;

	free(f);
}

void group_free(GROUP *g, int n)
{
	if(g==NULL)
		return;
	
	if(g->name!=NULL)
		free(g->name);
	
	if(g->faces!=NULL)
		free(g->faces);
	
	free(g);		
}

void model_free(MODEL *m)
{
	if(m==NULL)
		return;
	
	if(m->filename!=NULL)
		free(m->filename);

	group_free(m->groups, m->groups_n);
	face_free(m->faces, m->faces_n);
	material_free(m->mat, m->mat_n);

	if(m->draw_index!=NULL)
		free(m->draw_index);
	
	if(m->attr!=NULL)
		free(m->attr);
	
	if(m->v!=NULL)
		free(m->v);
	
	if(m->tex!=NULL)
		free(m->tex);
	
	if(m->norm!=NULL)
		free(m->norm);
	
	if(m->param!=NULL)
		free(m->param);
	
	free(m);
}


// recherches d'elements
int group_find(MODEL *model, char *name)
{
	int i;
	
	for(i= 0; i < model->groups_n; i++)
		if(strcmp(model->groups[i].name, name)==0)
			return i;
	
	return -1;
}

int material_find(MODEL *model, char *name)
{
	int i;
	
	for(i= 0; i < model->mat_n; i++)
		if(strcmp(model->mat[i].name, name)==0)
			return i;
	
	return -1;
}


int model_add_vertex(MODEL *model)
{
    assert(model != NULL);
    
    array_add(ARRAY(&model->v), &model->v_size, 
        model->v_n, 
        sizeof(VERTEX), MODEL_VERTEX_MORE);

    return model->v_n++;
}

int model_add_texcoord(MODEL *model)
{
    assert(model != NULL);
    
    array_add(ARRAY(&model->tex), &model->tex_size, 
        model->tex_n, 
        sizeof(VERTEXT), MODEL_VERTEX_MORE);
    
    return model->tex_n++;
}

int model_add_norm(MODEL *model)
{
    assert(model != NULL);
    
    array_add(ARRAY(&model->norm), &model->norm_size, 
        model->norm_n, 
        sizeof(VERTEXN), MODEL_VERTEX_MORE);

    return model->norm_n++;
}

int model_add_attr(MODEL *model, int v, int t, int n)
{
    assert(model != NULL);
    
    array_add(ARRAY(&model->attr), &model->attr_size, 
        model->attr_n, 
        sizeof(ATTR), MODEL_ATTR_MORE);

    model->attr[model->attr_n][gl_vertex]= v;
    model->attr[model->attr_n][gl_tex]= t;
    model->attr[model->attr_n][gl_norm]= n;
    
    return model->attr_n++;
}

int model_add_attr_mix(MODEL *model, int x, int y, float mix)
{
    assert(model != NULL);
    assert(x >= 0 && x < model->attr_n);
    assert(y >= 0 && y < model->attr_n);

    int v, t, n;
    
    v= -1;
    if(model_attr_get_vertex_id(model, x) >= 0 
    && model_attr_get_vertex_id(model, y) >= 0)
        v= model_add_vertex(model);
    assert(v >= 0);

    t= -1;
    if(model_attr_get_texcoord_id(model, x) >= 0 
    && model_attr_get_texcoord_id(model, y) >= 0)
        t= model_add_texcoord(model);

    n= -1;
    if(model_attr_get_norm_id(model, x) >= 0
    && model_attr_get_norm_id(model, y) >= 0)
        n= model_add_norm(model);
    
    vec3_mix(model_get_vertex_ptr(model, v),
        model_attr_get_vertex_ptr(model, x), model_attr_get_vertex_ptr(model, y), mix);
    if(t >= 0)
        vec2_mix(model_get_texcoord_ptr(model, t),
            model_attr_get_texcoord_ptr(model, x), model_attr_get_texcoord_ptr(model, y), mix);
    if(n >= 0)
        vec3_mix(model_get_norm_ptr(model, n),
            model_attr_get_norm_ptr(model, x), model_attr_get_norm_ptr(model, y), mix);
    
    return model_add_attr(model, v, t, n);
}

int model_add_face(MODEL *model, int norm, int mat)
{
    FACE *tmp;
    
    array_add(ARRAY(&model->faces), &model->faces_size, 
        model->faces_n, 
        sizeof(FACE), MODEL_FACE_MORE);
    
    tmp= &model->faces[model->faces_n];
    tmp->attr= model->attr_n;
    tmp->n= 0;
    tmp->norm= norm;
    tmp->mat= mat;

    return model->faces_n++;
}


inline void vertex_copy(VERTEX *d, VERTEX *v)
{
    vec4_copy((float *) d, (float *) v);
}

inline void tex_copy(VERTEXT *d, VERTEXT *t)
{
    vec2_copy((float *) d, (float *) t);
}

inline void norm_copy(VERTEXN *d, VERTEXN *n)
{
    vec4_copy((float *) d, (float *) n);
}

inline void attr_copy(ATTR *d, ATTR *a)
{
    (*d)[gl_vertex]= (*a)[gl_vertex];
    (*d)[gl_tex]= (*a)[gl_tex];
    (*d)[gl_norm]= (*a)[gl_norm];
}

inline void mat_copy(MATERIAL *d, MATERIAL *m)
{
    d->name= strdup(m->name);
}

int model_dup_attr(MODEL *model, int a)
{
    array_add(ARRAY(&model->attr), &model->attr_size, 
        model->attr_n, 
        sizeof(ATTR),  MODEL_ATTR_MORE);

    attr_copy(&model->attr[model->attr_n], &model->attr[a]);
    return model->attr_n++;
}

int model_dup_vertex(MODEL *model, int v)
{
    array_add(ARRAY(&model->v), &model->v_size, 
        model->v_n, 
        sizeof(VERTEX),  MODEL_VERTEX_MORE);

    vertex_copy(&model->v[model->v_n], &model->v[v]);
    return model->v_n++;
}

int model_dup_tex(MODEL *model, int t)
{
    array_add(ARRAY(&model->tex), &model->tex_size, 
        model->tex_n,
        sizeof(VERTEXT), MODEL_VERTEX_MORE);
    
    tex_copy(&model->tex[model->tex_n], &model->tex[t]);
    return model->tex_n++;
}

int model_dup_norm(MODEL *model, int n)
{
    array_add(ARRAY(&model->norm), &model->norm_size, 
        model->norm_n,
        sizeof(VERTEXN), MODEL_VERTEX_MORE);
    
    norm_copy(&model->norm[model->norm_n], &model->norm[n]);
    return model->norm_n++;
}

void face_set_triangle(MODEL *model, FACE *face, int a, int b, int c)
{
    int aa= model_dup_attr(model, a);
    int bb= model_dup_attr(model, b);
    assert(bb == aa + 1);
    int cc= model_dup_attr(model, c);
    assert(cc == aa + 2);

    face->attr= aa;
    face->n= 3;
}


static inline float inner_edges_get_area(VEC u, VEC v)
{
    VEC w;
    
    vec3_cross(w, u, v);
    return .5f * vec3_length(w);
}


float model_face_get_area(MODEL *model, FACE *face)
{
    VEC e0, e;
    float area;
    int i;

    assert(model!=NULL);
    assert(face!=NULL);
    
    area= .0f;
    model_face_get_inner_edge(model, face, 1, &e0);
    for(i= 2; i < face->n; i++)
    {
        model_face_get_inner_edge(model, face, i, &e);
        area+= inner_edges_get_area(e0, e);

        vec3_copy(e0, e);
    }
    
    return area;
}

float model_get_area(MODEL *model)
{
    float area= 0.f;
    int i, n;
    
    n= model_get_faces_n(model);
    for(i= 0; i < n; i++)
        area+= model_face_get_area(model, model_get_face_ptr(model, i));
    
    return area;
}


/* subdivise toutes les faces dont l'aire > area_max
 */

static float triangle_get_area(MODEL *model, int a, int b, int c)
{
    VEC e0, e1;
    float *v0, *v1, *v2;
    
    v0= model_attr_get_vertex_ptr(model, a);
    v1= model_attr_get_vertex_ptr(model, b);
    v2= model_attr_get_vertex_ptr(model, c);
    
    vec3_sub(e0, v1, v0);
    vec3_sub(e1, v2, v0);
    
    return inner_edges_get_area(e0, e1);
}

static void triangle_subdivide(MODEL *model, int a, int b, int c, int norm_id, int mat_id, float area_max)
{
    int ab, bc, ca;
    
    if(triangle_get_area(model, a, b, c) < area_max)
    {
        int sub_id= model_add_face(model, norm_id, mat_id);
        face_set_triangle(model, model_get_face_ptr(model, sub_id), a, b, c);
        return;
    }
    
    ab= model_add_attr_mix(model, a, b, .5f);
    bc= model_add_attr_mix(model, b, c, .5f);
    ca= model_add_attr_mix(model, c, a, .5f);

    triangle_subdivide(model, a, ab, ca, norm_id, mat_id, area_max);
    triangle_subdivide(model, ca, ab, bc, norm_id, mat_id, area_max);
    triangle_subdivide(model, bc, ab, b, norm_id, mat_id, area_max);
    triangle_subdivide(model, ca, bc, c, norm_id, mat_id, area_max);
}

int model_subdivide(MODEL **pmodel, float area_max)
{
    assert(pmodel != NULL);

    MODEL *model= *pmodel;
    assert(model != NULL);
    FACE *face;
    int a, b, c;
    int norm_id;
    int mat_id;
    int i, n;
    
    if((model->flags & MODEL_IS_TRIANGLES)==0)
    {
        model_set_triangles(pmodel);
        model= *pmodel;
    }
    
    n= model_get_faces_n(model);
    for(i= 0; i < n; i++)
    {
        face= model_get_face_ptr(model, i);
        norm_id= face_get_norm_id(face);
        mat_id= face_get_material_id(face);

        a= face_get_attr_id(face, 0);
        b= face_get_attr_id(face, 1);
        c= face_get_attr_id(face, 2);
        
        if(triangle_get_area(model, a, b, c) > area_max)
        {
            triangle_subdivide(model, a, b, c, norm_id, mat_id, area_max);
            
            // detruire la face subdivisee,
            model->faces_n--;
            model->faces[i]= model->faces[model->faces_n];
            // beurk, il va falloir re-ordonner les faces !
            // et eliminer tous les attributs non references
        }
    }
    
    return 1;
}


/* ajoute les normales geometriques des faces du modele
 */
void model_set_faces_norm(MODEL *m)
{
    ATTR *attr;
    int i;
    VEC u, v;
    
    printf("model '%s' : face normals ...\n", m->filename);

    array_add(ARRAY(&m->norm), &m->norm_size, 
        m->norm_n + m->faces_n -1,
        sizeof(VERTEXN), MODEL_NORM_MORE);
    
    for(i= 0; i<m->faces_n; i++)
    {
        if(m->faces[i].attr < 0)
            continue;

        m->faces[i].norm= m->norm_n;
        
        attr= &m->attr[m->faces[i].attr];
        vec3_sub(u, m->v[attr[1][gl_vertex]], m->v[attr[0][gl_vertex]]);
        vec3_sub(v, m->v[attr[m->faces[i].n -1][gl_vertex]], m->v[attr[0][gl_vertex]]);
        
        vec3_cross(m->norm[m->norm_n], u, v);
        vec3_norm(m->norm[m->norm_n], m->norm[m->norm_n]);
        
        m->norm_n++;
    }
    
    m->flags|= MODEL_HAS_FACES_NORM;
}

void model_set_vertex_norm(MODEL *model)
{
	FACE *face;
	int n;
	int i, j;
	
	printf("model '%s' : vertex normals ...\n", model->filename);

	if((model->flags & MODEL_HAS_FACES_NORM)==0)
		model_set_faces_norm(model);

	for(i= 0; i < model->faces_n; i++)
	{
		face= &model->faces[i];
		for(j= 0; j < face->n; j++)
		{	
			n= model->attr[face->attr + j][gl_norm];
			if(n < 0)
			{
				array_add(ARRAY(&model->norm), &model->norm_size,
					model->norm_n,
					sizeof(VERTEXN), MODEL_NORM_MORE);
				n= model->norm_n;
                            
				model->attr[face->attr + j][gl_norm]= model->norm_n;
				model->norm_n++;
			}

			vec3_zero(model->norm[n]);
		}
	}
	
	for(i= 0; i < model->faces_n; i++)
	{
		face= &model->faces[i];
		for(j= 0; j < face->n; j++)
		{	
			n= model->attr[face->attr + j][gl_norm];
			vec3_add(model->norm[n], model->norm[n], model->norm[face->norm]);
		}
	}

	for(i= 0; i < model->norm_n; i++)
	{
		vec3_norm(model->norm[i], model->norm[i]);
	}
	
	model->flags|= MODEL_HAS_NORM;
}

int model_is_triangles(MODEL *model)
{
	int n, i;
	
	if(model->flags & MODEL_IS_TRIANGLES)
		return 1;
	
	n= 0;
	for(i= 0; i < model->faces_n; i++)
	{
		if(model->faces[i].attr < 0)
			continue;
		
		if(model->faces[i].n != 3)
			n++;
	}

	if(n==0)
	{
		model->flags|= MODEL_IS_TRIANGLES;
		return 1;
	}
	
	model->flags&= ~MODEL_IS_TRIANGLES;
	return 0;
}

/* convertit toutes les faces (convexes) en triangles
 */
int model_set_triangles(MODEL **pmodel)
{
    MODEL *model;
    int i, j;
    int f, n;

    model= *pmodel;
    if(model==NULL)
            return -1;
    
    // triangulation du modele
    printf("model '%s' : triangulate ...\n", model->filename);
    
    n= model->faces_n;
    for(i= 0; i < n; i++)
    {
        if(model->faces[i].n==3 || model->faces[i].attr < 0)
            continue;
        
        for(j= 3; j < model->faces[i].n; j++)
        {
            f= model_add_face(model, model->faces[i].norm, model->faces[i].mat);
            
            face_set_triangle(model, &model->faces[f], 
                model->faces[i].attr,  		// face->vertex[0]
                model->faces[i].attr + j -1, 	// face->vertex[j-1]
                model->faces[i].attr + j);        // face->vertex[j]
        }
        
        model->faces[i].n= 3;
    }

    model->flags|= MODEL_IS_TRIANGLES;
    
    model_print_stat(model);
    return 1;
}


/* calcule la boite englobante du modele
 */
void model_set_aabox(MODEL *model)
{
	static const VERTEX vmin= {HUGE_VAL, HUGE_VAL, HUGE_VAL};
	static const VERTEX vmax= {-HUGE_VAL, -HUGE_VAL, -HUGE_VAL};
	
	FACE *face;
	VERTEX *v;
	int a;
	int i, j;
	
	vec3_copy(model->aabox[0], vmin);
	vec3_copy(model->aabox[1], vmax);
	
	for(i= 0; i < model->faces_n; i++)
	{
		face= &model->faces[i];
		for(j= 0; j < face->n; j++)
		{
			a= face->attr + j;
			v= &model->v[model->attr[a][gl_vertex]];
			
			if((*v)[0] < model->aabox[0][0])
				model->aabox[0][0]= (*v)[0];
			if((*v)[1] < model->aabox[0][1])
				model->aabox[0][1]= (*v)[1];
			if((*v)[2] < model->aabox[0][2])
				model->aabox[0][2]= (*v)[2];

			if((*v)[0] > model->aabox[1][0])
				model->aabox[1][0]= (*v)[0];
			if((*v)[1] > model->aabox[1][1])
				model->aabox[1][1]= (*v)[1];
			if((*v)[2] > model->aabox[1][2])
				model->aabox[1][2]= (*v)[2];
		}
	}
	
	printf("model '%s' : aabox (% -f % -f % -f) x (% -f % -f % -f)\n", 
		model->filename,
		model->aabox[0][0], model->aabox[0][1], model->aabox[0][2], 
		model->aabox[1][0], model->aabox[1][1], model->aabox[1][2]);
	
	model->flags|= MODEL_HAS_AABOX;
}

void model_set_scale(MODEL *model, float scale)
{
	VEC d;
	float w;
	int i;
	
	if((model->flags & MODEL_HAS_AABOX)==0)
		model_set_aabox(model);
	
	vec3_sub(d, model->aabox[1], model->aabox[0]);
	w= d[0];
	if(w < d[1])
		w= d[1];
	if(w < d[2])
		w= d[2];
	w= 2.f * scale / w;
	
	for(i= 0; i < model->v_n; i++)
		vec3_const_mul(model->v[i], w, model->v[i]);
}
