/*
    squelette application sdl
    
    mailto:jciehl@bat710.univ-lyon1.fr
    
    janvier 2007
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include <math.h>
#include <time.h>

#include <GL/glew.h>
#include <GL/glu.h>
#include <SDL/SDL.h>

#include "sdlkit.h"

#include "vec.h"
#include "model.h"
#include "maya_obj.h"
#include "off.h"

// position et orientation du point de vue
float camera_x, camera_y, camera_z;
float camera_angle, camera_angle2;
float camera_vitesse;

int lumiere;

// conversion degres vers radians
float radians(float d)
{
    return d * 0.0174532925f;
}

// affichage
void display(void (*display_func)(void *), void *display_param)
{
    // efface "l'ecran"
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    glMatrixMode(GL_MODELVIEW);	// selectionne la matrice de la scene
    glLoadIdentity();				// reinitialise les transformations

    if(lumiere!=0)
    {
        // active une source de lumiere et definit une matiere par defaut
        GLfloat mat_specular[] = { 1.0f, 1.0f, 1.0f, 1.0f };
        GLfloat mat_shininess[] = { 20.0f };
        GLfloat light_position[] = { 0.0f, 1000.0f, 100.0f, 0.0f };
        
        glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
        glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);
        glLightfv(GL_LIGHT0, GL_POSITION, light_position);
        
        glEnable(GL_LIGHTING);
        glEnable(GL_LIGHT0);
        glPolygonMode(GL_FRONT, GL_FILL);
        glPolygonMode(GL_BACK, GL_FILL);
    }
    else
    {
        // desactive la source de lumiere
        glDisable(GL_LIGHTING);
        glPolygonMode(GL_FRONT, GL_LINE);
        glPolygonMode(GL_BACK, GL_LINE);
    }
    
    // oriente la scene par rapport a la camera qui est restee en 0,0,0
    glRotatef(360.f - camera_angle2, 1.f, 0.f, 0.f);
    glRotatef(360.f - camera_angle, 0.f, 1.f, 0.f);
    glTranslatef(-camera_x, -camera_y, -camera_z);

    // desssine la scene
    display_func(display_param);
    
    SDL_GL_SwapBuffers();
}

void keyboard(void)
{
    SDLMod mod;
    unsigned char *keys;
    
    mod= SDL_GetModState();
    keys= sdlkit_get_keys();
    
    if(keys[SDLK_q] || keys[SDLK_ESCAPE])
        stop= 1;

    if(keys[SDLK_SPACE])
        lumiere= (lumiere +1) %2;

    // navigation
    if(keys[SDLK_UP])
    {
        if(mod&KMOD_CTRL || mod&KMOD_ALT)
        {
            // rotation de la camera sur X
            camera_angle2+= 1.f;
            if(camera_angle2 >= 360.f)
                    camera_angle2= 0.f;
        }
        else
        {
            // deplacement de la camera dans la direction de la visee
            camera_z-= cos(radians(camera_angle)) * camera_vitesse;
            camera_x-= sin(radians(camera_angle)) * camera_vitesse;
        }
    }
    
    if(keys[SDLK_DOWN])
    {
        if(mod&KMOD_CTRL || mod&KMOD_ALT)
        {
            // rotation de la camera sur Y+
            camera_angle2-= 1.f;
            if(camera_angle2 < 0.f)
                camera_angle2= 359.f;
        }
        else
        {
            // deplacement de la camera dans la direction opposee de la visee
            camera_z+= cos(radians(camera_angle)) * camera_vitesse;
            camera_x+= sin(radians(camera_angle)) * camera_vitesse;
        }
    }
    
    if(keys[SDLK_LEFT])
    {
        if(mod&KMOD_CTRL || mod&KMOD_ALT)
        {
            // rotation de la camera sur Y-
            camera_angle+= 1.f;
            if(camera_angle >= 360.f)
                camera_angle= 0.f;
        }
        else
        {
            // deplacement perpendiculaire a la direction de visee
            camera_z-= cos(radians(camera_angle + 90.f)) * camera_vitesse;
            camera_x-= sin(radians(camera_angle + 90.f)) * camera_vitesse;
        }
    }
    
    if(keys[SDLK_RIGHT])
    {
        if(mod&KMOD_CTRL || mod&KMOD_ALT)
        {
            // rotation de la camera sur Y+
            camera_angle-= 1.f;
            if(camera_angle < 0.f)
                camera_angle= 359.f;
        }
        else
        {
            // deplacement perpendiculaire a la direction de visee
            camera_z-= cos(radians(camera_angle - 90.f)) * camera_vitesse;
            camera_x-= sin(radians(camera_angle - 90.f)) * camera_vitesse;
        }
    }

    if(keys[SDLK_PAGEUP])
        camera_y+= camera_vitesse;
    
    if(keys[SDLK_PAGEDOWN])
        camera_y-= camera_vitesse;
}


// mise en place des parametres d'openGL
void initgl(void)
{
    glClearColor(0.0, 0.0, 0.0, 0.0);
    glClearDepth(1.);

    glShadeModel (GL_SMOOTH);
    
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_CULL_FACE);
    // glCullFace(GL_BACK);

    camera_x= 0.f;
    camera_y= 0.f;
    camera_z= 50.f;
    camera_angle= 0.f;
    camera_vitesse= 10.f;
    
    lumiere= 0;
    
    glMatrixMode(GL_PROJECTION);	// selectionne la matrice de visualisation
    glLoadIdentity();				// reinitialise les transformations

    gluPerspective(50., 1., 1., 10000.);
}


/* affichage du modele 
 */
void model_display_debug(MODEL *model)
{
    FACE *face;
    int i, n;

    glColor3f(.5f, .5f, .5f);
    glLineWidth(1.);
    n= model_get_faces_n(model);
    for(i= 0; i < n; i++)
    {
        face= model_get_face_ptr(model, i);
        if(face_get_vertex_n(face)==3)
        {
            glBegin(GL_TRIANGLES);
                glNormal3fv(model_face_get_norm_ptr(model, face));

                glVertex3fv(model_face_get_vertex_ptr(model, face, 0));
                glVertex3fv(model_face_get_vertex_ptr(model, face, 1));
                glVertex3fv(model_face_get_vertex_ptr(model, face, 2));
            glEnd();
        }
        else if(face_get_vertex_n(face)==4)
        {
            glBegin(GL_QUADS);
                glNormal3fv(model_face_get_norm_ptr(model, face));

                glVertex3fv(model_face_get_vertex_ptr(model, face, 0));
                glVertex3fv(model_face_get_vertex_ptr(model, face, 1));
                glVertex3fv(model_face_get_vertex_ptr(model, face, 2));
                glVertex3fv(model_face_get_vertex_ptr(model, face, 3));
            glEnd();
        }
    }
}

int main(int argc, char **argv)
{
    MODEL *model;
    
    srand48(time(NULL));

    if(argc != 2)
    {
        printf("usage: %s model.obj [width height]\n", argv[0]);
        printf("usage: %s model.off [width height]\n", argv[0]);
        return 0;
    }

    /* charge le modele .obj ou .off  */
    model= NULL;
    if(model_is_obj(argv[1])==0)
        model_load_obj(&model, argv[1]);
    if(model_is_off(argv[1])==0)
        model_load_off(&model, argv[1]);

    if(model==NULL)
    {
        printf("\n -- unable to load model\n");
        return 1;
    }
   
    /* ajouter les normales geometriques des faces */
    model_set_faces_norm(model);
    
    /* creation de la fenetre / contexte openGL */
    sdlkit_init(600, 600);
    
    /* visualisation interactive du modele */
    initgl();
    while(!sdlkit_stop())
    {
        sdlkit_events();
        keyboard();
        sdlkit_anime();
        display((void (*)(void *)) model_display_debug, (void *) model);
    }
   
    model_free(model);
    sdlkit_quit();
    printf("\ndone.\n");
    return 0;
}
