/*
    utilitaires GLSL
    
    mailto:jciehl@bat710.univ-lyon1.fr
    
    janvier 2007
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include <GL/glew.h>

#define VERBOSE 1
#define VERBOSE_DEBUG 1


void extFailed(char *message)
{
    printf("%-32s  [FAILED]\n", message);
    fflush(stdout);
}

void extSupported(char *message)
{
#if VERBOSE
    printf("%-32s  [  OK  ]\n", message);
    fflush(stdout);
#endif
}


int extIsSupported(char *ext, char *message, int verbose)
{
    if(glewIsSupported(ext))
    {
        if(verbose > 0)
            extSupported(message);
        return 0;
    }
    else
    {
#if VERBOSE_DEBUG	// n'affiche pas les tentatives
        if(verbose > 0)
            extFailed(message);
#endif
        return -1;	
    }
}

int glsl_load_source(char **str, int *n, char *fname)
{
    FILE *in;
    int size;
    
    *n= 0;
    *str= NULL;
    size= 0;		

    if(fname==NULL)
        return 0;
    in= fopen(fname, "r");
    if(in==NULL)
        return -1;

    do
    {		
        size+= 4096;
        *str= (char *) realloc(*str, size);
        assert(*str != NULL);

        (*n)+= (int) fread(*str + *n, 1, 4096, in);
    }
    while(!feof(in));
    
    fclose(in);
    (*str)[*n]= 0;
    return 0;
}


void glsl_program_info(GLuint program)
{
    char *log;
    int length= 0;

    glGetProgramiv(program, GL_INFO_LOG_LENGTH, &length);
    if(length > 0)
    {
        log= malloc(length+1);
        assert(log != NULL);
        
        glGetProgramInfoLog(program, length, &length, log);
        log[length]= 0;
        printf("\n%s", log);
        free(log);
    }
}

void glsl_shader_info(GLuint shader)
{
    char *log;
    int length= 0;

    glGetShaderiv(shader, GL_INFO_LOG_LENGTH, &length);
    if(length > 0)
    {
        log= malloc(length+1);
        assert(log != NULL);
        
        glGetShaderInfoLog(shader, length, &length, log);
        log[length]= 0;
        printf("\n%s", log);
        free(log);
    }
}

int glsl_add_shader(GLuint program, const GLchar *source, GLenum type)
{
    GLenum err;
    GLuint shader;
    GLint compiled;
    
    assert(program != 0);
    assert(type != 0);
    
    if(source==NULL)
        return 0;
    
    shader= glCreateShader(type);
    assert(shader != 0);
    glShaderSource(shader, 1, &source, NULL);
    glCompileShader(shader);

    compiled= GL_FALSE;
    glGetShaderiv(shader, GL_COMPILE_STATUS, &compiled);
    if(compiled==GL_FALSE)
    {
        extFailed("compile");
        glsl_shader_info(shader);
        return -1;
    }
#if VERBOSE_DEBUG
    else
        extSupported("compile");
#endif
    
    glAttachShader(program, shader);
    glDeleteShader(shader);
    
    err= glGetError();
    if(err != GL_NO_ERROR)
    {
        fprintf(stderr, "\nopenGL Error  %s:%d %s()\n%s\n", __FILE__, __LINE__, __func__, gluErrorString(err));
        return -1;
    }

    return 0;
}

GLuint glsl_program_init(char *vfname, char *ffname)
{
    GLchar *source;
    GLuint program;
    GLint linked;
    GLint validated;
    int length;
        
    program= glCreateProgram();
    
    // vertex shader
    if(glsl_load_source(&source, &length, vfname) < 0 
    || glsl_add_shader(program, source, GL_VERTEX_SHADER) < 0)
    {
        if(source!=NULL)
            free(source);
        extFailed("vertex shader");
        return 0;
    }
    else
    {
        if(source!=NULL)
            free(source);
        extSupported("vertex shader");
    }
    
    // fragment shader
    if(glsl_load_source(&source, &length, ffname) < 0 
    || glsl_add_shader(program, source, GL_FRAGMENT_SHADER) < 0)
    {
        if(source!=NULL)
            free(source);
        extFailed("fragment shader");
        return 0;
    }
    else
    {
        if(source!=NULL)
            free(source);
        extSupported("fragment shader");
    }

    // link
    glLinkProgram(program);
    linked= GL_FALSE;
    glGetProgramiv(program, GL_LINK_STATUS, &linked);
    if(linked==GL_FALSE)
    {
        extFailed("program link");
        glsl_program_info(program);
        return 0;
    }
#if VERBOSE_DEBUG
    else
        extSupported("program link");
#endif
    
    // validate
    glValidateProgram(program);
    validated= GL_FALSE;
    glGetProgramiv(program, GL_VALIDATE_STATUS, &validated);
    if(validated == GL_FALSE)
    {
        extFailed("program validation");
        glsl_program_info(program);
        return 0;
    }
#if VERBOSE_DEBUG
    else
        extSupported("program validation");
#endif
    
    // glUseProgram(program);

    return program;
}


int glsl_init(void)
{
    GLenum err;

    err= glewInit();
    if(err!=GLEW_OK)
    {
        printf("%s\n", glewGetErrorString(err));
        extFailed("GLEW");
        return -1;
    }
    
    if(!glewIsSupported("GL_VERSION_2_0"))
    {
        extFailed("OpenGL 2");
        return -1;
    }
    
#if VERBOSE
    extSupported("glew");
    extSupported("openGL 2");
#endif
    
    return 0;
}

int glsl_quit(void)
{
    return 0;
}

