/*
	scenes maya obj 3, ascii, triangulees
	
	mailto:jciehl@bat710.univ-lyon1.fr
	fevrier 2004 : premiere version
	fevrier 2006 : 
		stockage des attributs compatible avec vertex_buffer_objects
		triangulation du modele
		nettoyage du modele
*/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <math.h>

#include <GL/gl.h>

#include "model.h"
#include "vec.h"

#include "cull.h"

// tableaux dynamiques
void array_add(void **parray, int *size, int n, int e_size, int e_more)
{
	if(n >= *size)
	{
		*size= ((n / e_more) + 1) * e_more;
		
		*parray= realloc(*parray, *size * e_size);
		assert(*parray!=NULL);
	}
}

// gestion des structures
MODEL *model_new(char *filename)
{
	MODEL *m;
	
	m= malloc(sizeof(MODEL));	
	assert(m!=NULL);

#if 0
	char *s, *bs;
	
	s= strrchr(filename, '/');
	bs= strrchr(filename, '\\');
	if(s==NULL && bs!=NULL)
		s= bs;
	
	if(s!=NULL)
		m->filename= strdup(&s[1]);
	else
		m->filename= strdup(filename);

#else
	// fix: permet d'enregistrer les preferences dans le repertoire du modele
	m->filename= strdup(filename);
#endif

	assert(m->filename);

	m->flags= 0;
	
	m->groups= NULL;
	m->groups_size= 0;
	m->groups_n= 0;
	
	m->faces= NULL;
	m->faces_size= 0;
	m->faces_n= 0;
	
	m->draw_index= NULL;
	m->draw_index_n= 0;
	
	m->mat= NULL;
	m->mat_size= 0;
	m->mat_n= 0;
	
	m->attr= NULL;
	m->attr_size= 0;
	m->attr_n= 0;
	
	m->v= NULL;
	m->v_size= 0;
	m->v_n= 0;
	
	m->norm= NULL;
	m->norm_size= 0;
	m->norm_n= 0;
	
	m->tex= NULL;
	m->tex_size= 0;
	m->tex_n= 0;
	
	m->param= NULL;
	m->param_size= 0;
	m->param_n= 0;
	
	return m;
}

void model_print_stat(MODEL *model)
{
	int triangles_n, quads_n, polys_n;
	int i;
	
	printf("model '%s' : %d (%d) groups, %d (%d) faces, %d (%d) vertices\n", 
		model->filename, 
		model->groups_n, model->groups_size, 
		model->faces_n, model->faces_size, 
		model->v_n, model->v_size);

	triangles_n= 0;
	quads_n= 0;
	polys_n= 0;
	for(i= 0; i < model->faces_n; i++)
	{
		if(model->faces[i].attr < 0)
			continue;
		
		if(model->faces[i].n==3)
			triangles_n++;
		else if(model->faces[i].n==4)
			quads_n++;
		else
			polys_n++;
	}
	
	printf("%d triangles, %d quads, %d polygons\n", triangles_n, quads_n, polys_n);
	
	printf("materials %d (%d) :\n", model->mat_n, model->mat_size);
	for(i= 0; i < model->mat_n; i++)
		printf("  %s\n", model->mat[i].name);
}

void material_free(MATERIAL *m, int n)
{
	int i;
	
	if(m==NULL)
		return;
	
	for(i= 0; i < n; i++)
		if(m[i].name!=NULL)
			free(m[i].name);
	
	free(m);
}

void face_free(FACE *f, int n)
{
	if(f==NULL)
		return;

	free(f);
}

void group_free(GROUP *g, int n)
{
	if(g==NULL)
		return;
	
	if(g->name!=NULL)
		free(g->name);
	
	if(g->faces!=NULL)
		free(g->faces);
	
	free(g);		
}

void model_free(MODEL *m)
{
	if(m==NULL)
		return;
	
	if(m->filename!=NULL)
		free(m->filename);

	group_free(m->groups, m->groups_n);
	face_free(m->faces, m->faces_n);
	material_free(m->mat, m->mat_n);

	if(m->draw_index!=NULL)
		free(m->draw_index);
	
	if(m->attr!=NULL)
		free(m->attr);
	
	if(m->v!=NULL)
		free(m->v);
	
	if(m->tex!=NULL)
		free(m->tex);
	
	if(m->norm!=NULL)
		free(m->norm);
	
	if(m->param!=NULL)
		free(m->param);
	
	free(m);
}


// recherches d'elements
int group_find(MODEL *model, char *name)
{
	int i;
	
	for(i= 0; i < model->groups_n; i++)
		if(strcmp(model->groups[i].name, name)==0)
			return i;
	
	return -1;
}

int material_find(MODEL *model, char *name)
{
	int i;
	
	for(i= 0; i < model->mat_n; i++)
		if(strcmp(model->mat[i].name, name)==0)
			return i;
	
	return -1;
}


inline void vertex_copy(VERTEX *d, VERTEX *v)
{
	vec_copy(*d, *v);
}

inline void tex_copy(VERTEXT *d, VERTEXT *t)
{
	vec2_copy(*d, *t);
}

inline void norm_copy(VERTEXN *d, VERTEXN *n)
{
	vec_copy(*d, *n);
}

inline void attr_copy(ATTR *d, ATTR *a)
{
	vec3_copy(*d, *a);
}

inline void mat_copy(MATERIAL *d, MATERIAL *m)
{
	d->name= strdup(m->name);
}

int model_add_face(MODEL *model, int norm, int mat)
{
	FACE *tmp;
	
	array_add((void **) &model->faces, &model->faces_size, model->faces_n, 
		sizeof(FACE), MODEL_FACE_MORE);
	
	tmp= &model->faces[model->faces_n];
	tmp->attr= model->attr_n;
	tmp->n= 0;
	tmp->norm= norm;
	tmp->mat= mat;
	return model->faces_n++;
}

int model_dup_attr(MODEL *model, int a)
{
	array_add((void **) &model->attr, &model->attr_size, model->attr_n, 
		sizeof(ATTR),  MODEL_ATTR_MORE);

	attr_copy(&model->attr[model->attr_n], &model->attr[a]);
	return model->attr_n++;
}

int model_dup_vertex(MODEL *model, int v)
{
	array_add((void **) &model->v, &model->v_size, model->v_n, 
		sizeof(VERTEX),  MODEL_VERTEX_MORE);

	vertex_copy(&model->v[model->v_n], &model->v[v]);
	return model->v_n++;
}

int model_dup_tex(MODEL *model, int t)
{
	array_add((void **) &model->tex, &model->tex_size, model->tex_n,
		sizeof(VERTEXT), MODEL_VERTEX_MORE);
	
	tex_copy(&model->tex[model->tex_n], &model->tex[t]);
	return model->tex_n++;
}

int model_dup_norm(MODEL *model, int n)
{
	array_add((void **) &model->norm, &model->norm_size, model->norm_n,
		sizeof(VERTEXN), MODEL_VERTEX_MORE);
	
	norm_copy(&model->norm[model->norm_n], &model->norm[n]);
	return model->norm_n++;
}

void face_set_triangle(MODEL *model, FACE *face, int a, int b, int c)
{
	face->attr= model->attr_n;

	model_dup_attr(model, a);
	model_dup_attr(model, b);
	model_dup_attr(model, c);
	face->n= 3;
}

float model_face_get_area(MODEL *model, int f)
{
	VEC norm;
	VEC u, v;
	FACE *face;
	ATTR *attr;
	
	face= &model->faces[f];
	attr= &model->attr[face->attr];
	vec3_sub(u, model->v[attr[1][gl_vertex]], model->v[attr[0][gl_vertex]]);
	vec3_sub(v, model->v[attr[2][gl_vertex]], model->v[attr[0][gl_vertex]]);
	
	vec3_cross(norm, u, v);
	return .5f * vec3_length(norm);
}


/* ajoute les normales geometriques des faces du modele
 */
void model_set_faces_norm(MODEL *m)
{
	ATTR *attr;
	int i;
	VEC u, v;
	
	printf("model '%s' : face normals ...\n", m->filename);

	array_add((void **) &m->norm, &m->norm_size, m->norm_n + m->faces_n -1,
		sizeof(VERTEXN), MODEL_NORM_MORE);
	
	for(i= 0; i<m->faces_n; i++)
	{
		if(m->faces[i].attr < 0)
			continue;

		m->faces[i].norm= m->norm_n;
		
		attr= &m->attr[m->faces[i].attr];
		vec3_sub(u, m->v[attr[1][gl_vertex]], m->v[attr[0][gl_vertex]]);
		vec3_sub(v, m->v[attr[m->faces[i].n -1][gl_vertex]], m->v[attr[0][gl_vertex]]);
		
		vec3_cross(m->norm[m->norm_n], u, v);
		vec3_norm(m->norm[m->norm_n], m->norm[m->norm_n]);
		
		m->norm_n++;
	}
	
	m->flags|= MODEL_HAS_FACES_NORM;
}

void model_set_vertex_norm(MODEL *model)
{
	FACE *face;
	int n;
	int i, j;
	
	printf("model '%s' : vertex normals ...\n", model->filename);

	if((model->flags & MODEL_HAS_FACES_NORM)==0)
		model_set_faces_norm(model);

	for(i= 0; i < model->faces_n; i++)
	{
		face= &model->faces[i];
		for(j= 0; j < face->n; j++)
		{	
			n= model->attr[face->attr + j][gl_norm];
			if(n < 0)
			{
				array_add((void **) &model->norm, &model->norm_size,
					model->norm_n,
					sizeof(VERTEXN), MODEL_NORM_MORE);
				n= model->norm_n;
				model->attr[face->attr + j][gl_norm]= model->norm_n;
				model->norm_n++;
			}

			vec3_zero(model->norm[n]);
		}
	}
	
	for(i= 0; i < model->faces_n; i++)
	{
		face= &model->faces[i];
		for(j= 0; j < face->n; j++)
		{	
			n= model->attr[face->attr + j][gl_norm];
			vec3_add(model->norm[n], model->norm[n], model->norm[face->norm]);
		}
	}

	for(i= 0; i < model->norm_n; i++)
	{
		vec3_norm(model->norm[i], model->norm[i]);
	}
	
	model->flags|= MODEL_HAS_NORM;
}

int model_is_triangles(MODEL *model)
{
	int n, i;
	
	if(model->flags & MODEL_IS_TRIANGLES)
		return 1;
	
	n= 0;
	for(i= 0; i < model->faces_n; i++)
	{
		if(model->faces[i].attr < 0)
			continue;
		
		if(model->faces[i].n != 3)
			n++;
	}

	if(n==0)
	{
		model->flags|= MODEL_IS_TRIANGLES;
		return 1;
	}
	
	model->flags&= ~MODEL_IS_TRIANGLES;
	return 0;
}

/* convertit toutes les faces (convexes) en triangles
 */
int model_set_triangles(MODEL **pmodel)
{
	MODEL *model;
	int i, j;
	int f, n;

	model= *pmodel;
	if(model==NULL)
		return -1;
	
	// triangulation du modele
	printf("model '%s' : triangulate ...\n", model->filename);
	
	n= model->faces_n;
	for(i= 0; i < n; i++)
	{
		if(model->faces[i].n==3 || model->faces[i].attr < 0)
			continue;
		
		for(j= 3; j < model->faces[i].n; j++)
		{
			f= model_add_face(model, model->faces[i].norm, model->faces[i].mat);
			
			face_set_triangle(model, &model->faces[f], 
				model->faces[i].attr,  		// face->vertex[0]
				model->faces[i].attr + j-1, 	// face->vertex[j-1]
				model->faces[i].attr + j);	// face->vertex[j]
		}
		
		model->faces[i].n= 3;
	}

	model->flags|= MODEL_IS_TRIANGLES;
	
	model_print_stat(model);
	return 1;
}


/* calcule la boite englobante du modele
 */
void model_set_aabox(MODEL *model)
{
	static const VERTEX vmin= {HUGE_VAL, HUGE_VAL, HUGE_VAL};
	static const VERTEX vmax= {-HUGE_VAL, -HUGE_VAL, -HUGE_VAL};
	
	FACE *face;
	VERTEX *v;
	int a;
	int i, j;
	
	vec3_copy(model->aabox[0], vmin);
	vec3_copy(model->aabox[1], vmax);
	
	for(i= 0; i < model->faces_n; i++)
	{
		face= &model->faces[i];
		for(j= 0; j < face->n; j++)
		{
			a= face->attr + j;
			v= &model->v[model->attr[a][gl_vertex]];
			
			if((*v)[0] < model->aabox[0][0])
				model->aabox[0][0]= (*v)[0];
			if((*v)[1] < model->aabox[0][1])
				model->aabox[0][1]= (*v)[1];
			if((*v)[2] < model->aabox[0][2])
				model->aabox[0][2]= (*v)[2];

			if((*v)[0] > model->aabox[1][0])
				model->aabox[1][0]= (*v)[0];
			if((*v)[1] > model->aabox[1][1])
				model->aabox[1][1]= (*v)[1];
			if((*v)[2] > model->aabox[1][2])
				model->aabox[1][2]= (*v)[2];
		}
	}
	
	printf("model '%s' : aabox (% -f % -f % -f) x (% -f % -f % -f)\n", 
		model->filename,
		model->aabox[0][0], model->aabox[0][1], model->aabox[0][2], 
		model->aabox[1][0], model->aabox[1][1], model->aabox[1][2]);
	
	model->flags|= MODEL_HAS_AABOX;
}

void model_set_scale(MODEL *model, float scale)
{
	VEC d;
	float w;
	int i;
	
	if((model->flags & MODEL_HAS_AABOX)==0)
		model_set_aabox(model);
	
	vec3_sub(d, model->aabox[1], model->aabox[0]);
	w= d[0];
	if(w < d[1])
		w= d[1];
	if(w < d[2])
		w= d[2];
	w= 2.f * scale / w;
	
	for(i= 0; i < model->v_n; i++)
		vec3_const_mul(model->v[i], w, model->v[i]);
}

/* affichage du modele 
 */
void model_display_debug(MODEL *model)
{
	ATTR *iv;
	int i;

	glLineWidth(1.);
	for(i= 0; i < model->faces_n; i++)
	{
		if(model->faces[i].attr < 0)
			continue;
		
		iv= &model->attr[model->faces[i].attr];
		
		if(strcmp(model->mat[model->faces[i].mat].name, "diffuseLuminaire1SG")==0)
			// sources de lumieres en jaune
			glColor3f(.5f, .5f, 0.f);
		else
			// objets en blanc
			glColor3f(.5f, .5f, .5f);

		if(model->faces[i].n==3)
		{
			// glBegin(GL_LINE_LOOP);
			glBegin(GL_TRIANGLES);
			
			glNormal3fv(model->norm[model->faces[i].norm]);
			
			glVertex3fv(model->v[iv[0][gl_vertex]]);
			glVertex3fv(model->v[iv[1][gl_vertex]]);
			glVertex3fv(model->v[iv[2][gl_vertex]]);
			glEnd();
		}
		else if(model->faces[i].n==4)
		{
			printf("quad %d (attr %d)\n", i, model->faces[i].attr);
			
			glColor3f(1.f, 0.f, 0.f);
			// glBegin(GL_LINE_LOOP);
			glBegin(GL_QUADS);

			glNormal3fv(model->norm[model->faces[i].norm]);

			glVertex3fv(model->v[iv[0][gl_vertex]]);
			glVertex3fv(model->v[iv[1][gl_vertex]]);
			glVertex3fv(model->v[iv[2][gl_vertex]]);
			glVertex3fv(model->v[iv[3][gl_vertex]]);
			glEnd();
		}
	}
}


void model_display_init(MODEL *model)
{
	int i;
	
	if(model->draw_index!=NULL)
		return;
	
	model->draw_index= malloc(sizeof(int) * model->attr_n);
	assert(model->draw_index!=NULL);
	
	for(i= 0; i < model->attr_n; i++)
		model->draw_index[i]= model->attr[i][gl_vertex];
	
	model->draw_index_n= model->attr_n;
}	

void model_display(MODEL *model)
{
	glLineWidth(1.f);

	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3, GL_FLOAT, sizeof(VERTEX), model->v);
	
	if((model->flags & MODEL_HAS_NORM)!=0)
	{
		glEnableClientState(GL_NORMAL_ARRAY);
		glNormalPointer(GL_FLOAT, sizeof(VERTEXN), model->norm);
	}
	else
		glDisableClientState(GL_NORMAL_ARRAY);
	
	glDrawElements(GL_TRIANGLES, model->draw_index_n, 
		GL_UNSIGNED_INT, model->draw_index);
}

void model_cull_display(MODEL *model)
{
	CULL cull;
	cull_set(&cull);

	if(!(model->flags & MODEL_HAS_AABOX))
		model_set_aabox(model);
	
	// if(cull_vertex(&cull, model->v[0]))
	if(cull_aabox(&cull, model->aabox))
		glColor3f(1.f, 1.f, 1.f);
	else
		glColor3f(1.f, 0.f, 0.f);
	
	glLineWidth(1.f);

	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3, GL_FLOAT, sizeof(VERTEX), model->v);
	
	if((model->flags & MODEL_HAS_NORM)!=0)
	{
		glEnableClientState(GL_NORMAL_ARRAY);
		glNormalPointer(GL_FLOAT, sizeof(VERTEXN), model->norm);
	}
	else
		glDisableClientState(GL_NORMAL_ARRAY);
	
	glDrawElements(GL_TRIANGLES, model->draw_index_n, 
		GL_UNSIGNED_INT, model->draw_index);

{
	VERTEX v;
	int i, p;

	glPointSize(10.f);
	glColor3f(1.f, 1.f, 0.f);
	glBegin(GL_POINTS);

	for(i= 0; i < 8; i++)
	{
		// genere les sommets de la boite englobante a partir des extremes
		for(p= 0; p < 3; p++)
		{
			if(i & (1<<p))
				v[p]= model->aabox[1][p];
			else
				v[p]= model->aabox[0][p];
		}
		glVertex3fv(v);
	}
	
	glEnd();
}
}
