#include <stdio.h>
#include <GL/gl.h>
#include <math.h>

#include "vec.h"
#include "glmat.h"

void glmat_identity(GLfloat m[16])
{
	m[0+4*0] = 1; m[0+4*1] = 0; m[0+4*2] = 0; m[0+4*3] = 0;
	m[1+4*0] = 0; m[1+4*1] = 1; m[1+4*2] = 0; m[1+4*3] = 0;
	m[2+4*0] = 0; m[2+4*1] = 0; m[2+4*2] = 1; m[2+4*3] = 0;
	m[3+4*0] = 0; m[3+4*1] = 0; m[3+4*2] = 0; m[3+4*3] = 1;
}

int glmat_invert(const GLfloat src[16], GLfloat inverse[16])
{
	int i, j, k, swap;
	float t;
	GLfloat temp[4][4];

	for (i=0; i<4; i++) {
		for (j=0; j<4; j++) {
		    temp[i][j] = src[i*4+j];
		}
	}

	glmat_identity(inverse);

	for (i = 0; i < 4; i++)
	{
		/*
		** Look for largest element in column
		*/
		swap = i;
		for (j = i + 1; j < 4; j++)
		{
			if (fabs(temp[j][i]) > fabs(temp[i][i]))
				swap = j;
		}

		if (swap != i)
		{
			/*
			** Swap rows.
			*/
			for (k = 0; k < 4; k++)
			{
				t = temp[i][k];
				temp[i][k] = temp[swap][k];
				temp[swap][k] = t;
		
				t = inverse[i*4+k];
				inverse[i*4+k] = inverse[swap*4+k];
				inverse[swap*4+k] = t;
			}
		}

		if (temp[i][i] == 0.0f)
		{
			/*
			** No non-zero pivot.  The matrix is singular, which shouldn't
			** happen.  This means the user gave us a bad matrix.
			*/
			return GL_FALSE;
		}

		t = temp[i][i];
		for (k = 0; k < 4; k++)
		{
			temp[i][k] /= t;
			inverse[i*4+k] /= t;
		}
		
		for (j = 0; j < 4; j++)
		{
			if (j != i)
			{
				t = temp[j][i];
				for (k = 0; k < 4; k++)
				{
					temp[j][k] -= temp[i][k]*t;
					inverse[j*4+k] -= inverse[i*4+k]*t;
				}
			}
		}
	}

	return GL_TRUE;
}

void glmat_mul(const GLfloat a[16], const GLfloat b[16], GLfloat result[16])
{
	int i, j;

	for (i = 0; i < 4; i++) {
		for (j = 0; j < 4; j++) {
			result[i*4+j] = 
				a[i*4+0]*b[0*4+j] +
				a[i*4+1]*b[1*4+j] +
				a[i*4+2]*b[2*4+j] +
				a[i*4+3]*b[3*4+j];
		}
	}
}



void glvec_normalize(GLfloat v[3])
{
	vec3_norm(v, v);
}

void glvec_cross(GLfloat v1[3], GLfloat v2[3], GLfloat result[3])
{
	vec3_cross(result, v1, v2);
}

void glmat_mul_vec(const GLfloat matrix[16], const GLfloat v[4], GLfloat result[4])
{
	GLfloat tmp[4];
	int i;

	for (i=0; i<4; i++) {
		tmp[i] = 
			v[0]*matrix[0*4+i] +
			v[1]*matrix[1*4+i] +
			v[2]*matrix[2*4+i] +
			v[3]*matrix[3*4+i];
	}

	vec_copy(result, tmp);	
}


void glvec_print(GLVEC v, char *m)
{
	printf("%s=\t(%- .4f %- .4f %- .4f %- .4f)\n", m,
		v[0], v[1], v[2], v[3]);
}

void glmat_print(GLMAT a, char *m)
{
	printf("%s=\t(%- .4f %- .4f %- .4f %- .4f\n", m,
		a[0], a[1], a[2], a[3]);
	printf("\t %- .4f %- .4f %- .4f %- .4f\n",
		a[4], a[5], a[6], a[7]);
	printf("\t %- .4f %- .4f %- .4f %- .4f\n",
		a[8], a[9], a[10], a[11]);
	printf("\t %- .4f %- .4f %- .4f %- .4f)\n\n",
		a[12], a[13], a[14], a[15]);
}
